package ru.yandex.direct.internaltools.core;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Базовый класс, от которого должны отнаследываться все реализации внутренних инструментов
 *
 * @param <T> тип входных параметров класса
 */
@ParametersAreNonnullByDefault
public interface BaseInternalTool<T extends InternalToolParameter> {
    /**
     * Произвести валидацию входящих параметров. Эта валидация вызывается после валидации уровня приложения, поэтому,
     * если какое-то поле было отмечено как необходимое, можно уже не проверять здесь, что оно null или пустое.
     * Приложение
     * гарантирует что такое поле не дойдет до инструмента
     * <p>
     * Не стоит как-то изменять ввод или что-то сохранять в классе в процессе валидации, так как класс инструмента
     * в приложении существует только в одном экземпляре и может быть вызван из нескольких потоков
     *
     * @param t те самые параметры, которые позже будут переданы методу process этого же класса.
     * @return результат валидации, пустой или с ошибками
     */
    default ValidationResult<T, Defect> validate(T t) {
        // Can be implemented in subclass
        return ItemValidationBuilder.<T, Defect>of(t).getResult();
    }

    /**
     * Выполнить работу инструмента без ввода.
     * <p>
     * Результат работы метода отдается пользователю при начале работы с инструментом вместе с описанием самого
     * инструмента.
     * Может быть использовано для передачи информации об изначальном состоянии системы. В этом методе не стоит
     * выполнять
     * пишущих действий
     */
    default InternalToolResult processWithoutInput() {
        return null;
    }

    /**
     * Выполнить основную полезную работу инструмента с заданным вводом
     */
    InternalToolResult process(T t);

    /**
     * Выполнить преобразования над билдером обертки над инструментом, если это нужно.
     * Вызывается один раз перед тем как обертка будет создана и добавлена в каталог инструментов.
     *
     * @param builder полностью заполненный, билдер обертки над инструментом
     * @return новый экземпляр заполненного билдера и модифицированный переданный билдер
     */
    default InternalToolProxy.Builder<T> preCreate(InternalToolProxy.Builder<T> builder) {
        return builder;
    }
}
