package ru.yandex.direct.internaltools.core;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.container.InternalToolDetailsData;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enrich.InternalToolEnrichProcessor;
import ru.yandex.direct.internaltools.core.enrich.InternalToolEnrichProcessorFactory;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolProcessingException;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.input.InternalToolInputGroup;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.internaltools.core.bootstrap.InternalToolProxyBootstrap.extractFileFieldsNames;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

/**
 * Надстройка над объектов внутреннего инструмента. Содержит в себе все его настройки и параметры в развернутом виде и умеет правильно его запускать
 *
 * @param <T>
 */
@ParametersAreNonnullByDefault
public class InternalToolProxy<T extends InternalToolParameter> {
    public static class Builder<T extends InternalToolParameter> {
        private BaseInternalTool<T> internalTool;
        private Set<InternalToolAccessRole> allowedRoles;
        private FeatureName requiredFeature;
        private InternalToolCategory category;
        private String label;
        private String name;
        private String description;
        private boolean acceptsFiles;
        private InternalToolAction action;
        private InternalToolType type;
        private List<String> disclaimers;
        private Class<T> inputClass;
        private List<InternalToolInputGroup<T>> inputGroups;
        private InternalToolEnrichProcessorFactory enrichProcessorFactory;

        public InternalToolProxy<T> build() {
            return new InternalToolProxy<>(internalTool, allowedRoles, requiredFeature, category, label, name,
                    description, action, type, disclaimers, inputClass, inputGroups, enrichProcessorFactory,
                    extractFileFieldsNames(inputGroups));
        }

        public BaseInternalTool<T> getInternalTool() {
            return internalTool;
        }

        public Builder<T> withInternalTool(BaseInternalTool<T> internalTool) {
            this.internalTool = internalTool;
            return this;
        }

        public Set<InternalToolAccessRole> getAllowedRoles() {
            return allowedRoles;
        }

        public Builder<T> withAllowedRoles(Set<InternalToolAccessRole> allowedRoles) {
            this.allowedRoles = allowedRoles;
            return this;
        }

        public Builder<T> withRequiredFeature(@Nullable FeatureName requiredFeature) {
            this.requiredFeature = requiredFeature;
            return this;
        }

        public InternalToolCategory getCategory() {
            return category;
        }

        public Builder<T> withCategory(InternalToolCategory category) {
            this.category = category;
            return this;
        }

        public String getLabel() {
            return label;
        }

        public Builder<T> withLabel(String label) {
            this.label = label;
            return this;
        }

        public String getName() {
            return name;
        }

        public Builder<T> withName(String name) {
            this.name = name;
            return this;
        }

        public String getDescription() {
            return description;
        }

        public Builder<T> withDescription(String description) {
            this.description = description;
            return this;
        }

        public boolean isAcceptsFiles() {
            return acceptsFiles;
        }

        public Builder<T> withAcceptsFiles(boolean acceptsFiles) {
            this.acceptsFiles = acceptsFiles;
            return this;
        }

        public InternalToolAction getAction() {
            return action;
        }

        public Builder<T> withAction(InternalToolAction action) {
            this.action = action;
            return this;
        }

        public InternalToolType getType() {
            return type;
        }

        public Builder<T> withType(InternalToolType type) {
            this.type = type;
            return this;
        }

        public List<String> getDisclaimers() {
            return disclaimers;
        }

        public Builder<T> withDisclaimers(List<String> disclaimers) {
            this.disclaimers = disclaimers;
            return this;
        }

        public Class<T> getInputClass() {
            return inputClass;
        }

        public Builder<T> withInputClass(Class<T> inputClass) {
            this.inputClass = inputClass;
            return this;
        }

        public List<InternalToolInputGroup<T>> getInputGroups() {
            return inputGroups;
        }

        public Builder<T> withInputGroups(List<InternalToolInputGroup<T>> inputGroups) {
            this.inputGroups = inputGroups;
            return this;
        }

        public InternalToolEnrichProcessorFactory getEnrichProcessorFactory() {
            return enrichProcessorFactory;
        }

        public Builder<T> withEnrichProcessorFactory(
                InternalToolEnrichProcessorFactory enrichProcessorFactory) {
            this.enrichProcessorFactory = enrichProcessorFactory;
            return this;
        }
    }

    private final BaseInternalTool<T> internalTool;
    private final Set<InternalToolAccessRole> allowedRoles;
    private final FeatureName requiredFeature;
    private final InternalToolCategory category;
    private final String label;
    private final String name;
    private final String description;
    private final boolean acceptsFiles;
    private final InternalToolAction action;
    private final InternalToolType type;
    private final List<String> disclaimers;
    private final Class<T> inputClass;
    private final List<InternalToolInputGroup<T>> inputGroups;
    private final InternalToolEnrichProcessorFactory enrichProcessorFactory;
    private final Set<String> fileFieldsNames;
    private List<InternalToolEnrichProcessor> enrichProcessors;

    public InternalToolProxy(
            BaseInternalTool<T> internalTool,
            Set<InternalToolAccessRole> allowedRoles,
            @Nullable FeatureName requiredFeature,
            InternalToolCategory category,
            String label,
            String name,
            String description,
            InternalToolAction action,
            InternalToolType type,
            List<String> disclaimers,
            Class<T> inputClass,
            List<InternalToolInputGroup<T>> inputGroups,
            InternalToolEnrichProcessorFactory enrichProcessorFactory,
            Set<String> fileFieldsNames) {
        this.internalTool = internalTool;
        this.allowedRoles = allowedRoles;
        this.requiredFeature = requiredFeature;
        this.category = category;
        this.label = label;
        this.name = name;
        this.description = description;
        this.action = action;
        this.type = type;
        this.disclaimers = disclaimers;
        this.inputClass = inputClass;
        this.inputGroups = inputGroups;
        this.enrichProcessorFactory = enrichProcessorFactory;
        this.fileFieldsNames = fileFieldsNames;
        this.acceptsFiles = !fileFieldsNames.isEmpty();
    }

    /**
     * Запустить обработчик инструмента по-умолчанию
     */
    public InternalToolResult bareRun() {
        InternalToolResult result = internalTool.processWithoutInput();
        addDetails(result);
        return result;
    }

    /**
     * Запустить основной обработчик инструмента с переданными параметрами, предварительно их провалидировав
     */
    public InternalToolResult process(Map<String, Object> inputParam, User operator) {
        T param;
        try {
            param = JsonUtils.getObjectMapper().convertValue(inputParam, inputClass);
        } catch (IllegalArgumentException e) {
            throw new InternalToolValidationException("Got unparsable input");
        }
        param.setOperator(operator);

        ValidationResult<T, Defect> validationResult;
        try {
            validationResult = validateParam(param);
        } catch (InternalToolValidationException e) {
            throw e;
        } catch (Exception e) {
            throw new InternalToolProcessingException("Unexpected error when validating input", e);
        }

        if (validationResult == null || validationResult.hasAnyErrors()) {
            throw new InternalToolValidationException("Got invalid input").withValidationResult(validationResult);
        }

        InternalToolResult result;
        try {
            result = internalTool.process(param);
        } catch (InternalToolProcessingException | InternalToolValidationException e) {
            throw e;
        } catch (Exception e) {
            throw new InternalToolProcessingException(e.getMessage(), e);
        }

        try {
            addDetails(result);
        } catch (Exception e) {
            throw new InternalToolProcessingException("Unexpected error when adding rich data to fields", e);
        }
        return result;
    }

    /**
     * Провалидировать параметры вызова и вернуть результат валидации.
     * <p>
     * Сначала вызывается общая валидация, затем, в случае если она прошла, валидация, заданная внутри интрумента
     */
    private ValidationResult<T, Defect> validateParam(T param) {
        ValidationResult<T, Defect> validationResult = preValidateParam(param);
        if (validationResult.hasAnyErrors()) {
            return validationResult;
        }

        return internalTool.validate(param);
    }

    private ValidationResult<T, Defect> preValidateParam(T param) {
        ItemValidationBuilder<T, Defect> validationBuilder = ItemValidationBuilder.of(param);
        validationBuilder
                .check(notNull());

        inputGroups.forEach(g -> g.addValidation(validationBuilder, param));

        return validationBuilder.getResult();
    }

    /**
     * Добавить подробную информацию про некоторые ключи, если это нужно.
     * Работает только с результатом в виде класса InternalToolMassResult
     */
    private void addDetails(InternalToolResult result) {
        if (!(result instanceof InternalToolMassResult)) {
            return;
        }

        InternalToolMassResult<?> massResult = (InternalToolMassResult<?>) result;
        List<?> data = massResult.getData();
        if (data.isEmpty()) {
            return;
        }

        List<InternalToolEnrichProcessor> processors = getInternalToolEnrichProcessors(data);

        Map<String, InternalToolDetailsData> details = new HashMap<>();
        for (InternalToolEnrichProcessor processor : processors) {
            details.put(processor.getFieldName(), processor.fetchDetails(data));
        }

        massResult.setDetails(details);
    }

    private List<InternalToolEnrichProcessor> getInternalToolEnrichProcessors(List<?> data) {
        if (enrichProcessors == null) {
            enrichProcessors = createEnrichProcessors(data.get(0).getClass());
        }
        return enrichProcessors;
    }

    private List<InternalToolEnrichProcessor> createEnrichProcessors(Class<?> cls) {
        return Arrays.stream(cls.getDeclaredFields())
                .filter(f -> !f.getName().contains("$"))    // filter dynamically enhanced fields
                .map(f -> enrichProcessorFactory.forField(f, cls))
                .filter(Objects::nonNull)
                .collect(toList());
    }

    public String getLabel() {
        return label;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public InternalToolCategory getCategory() {
        return category;
    }

    public InternalToolAction getAction() {
        return action;
    }

    public List<String> getDisclaimers() {
        return disclaimers;
    }

    public InternalToolType getType() {
        return type;
    }

    public boolean writesData() {
        return type != InternalToolType.REPORT;
    }

    public List<InternalToolInputGroup<T>> getInputGroups() {
        return inputGroups;
    }

    public Set<InternalToolAccessRole> getAllowedRoles() {
        return EnumSet.copyOf(allowedRoles);
    }

    public FeatureName getRequiredFeature() {
        return requiredFeature;
    }

    public BaseInternalTool<T> getInternalTool() {
        return internalTool;
    }

    public boolean isAcceptsFiles() {
        return acceptsFiles;
    }

    public Set<String> getFileFieldsNames() {
        return fileFieldsNames;
    }
}
