package ru.yandex.direct.internaltools.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.exception.InternalToolAccessDeniedException;
import ru.yandex.direct.internaltools.core.exception.InternalToolNotFoundException;


/**
 * Каталог всех внутренних инструментов. Умеет отдавать списки категорий и экземпляры отчетов, предварительно проверяя
 * права на них.
 */
@ParametersAreNonnullByDefault
public class InternalToolsRegistry {

    private final Map<String, InternalToolProxy<?>> labelToTool;
    private final FeatureService featureService;

    public InternalToolsRegistry(Map<String, InternalToolProxy<?>> labelToTool, FeatureService featureService) {
        this.labelToTool = labelToTool;
        this.featureService = featureService;
    }

    /**
     * Получить объект-обертку над внутренним инструментом
     *
     * @param label        уникальный строковый идентификатор инструмента
     * @param accessLevels уровни доступа запрашивающего отчет пользователя
     */
    public InternalToolProxy getInternalToolProxy(String label, Collection<InternalToolAccessRole> accessLevels,
                                                  ClientId operatorClientId) {
        if (!labelToTool.containsKey(label)) {
            throw new InternalToolNotFoundException(String.format("Tool %s not found", label));
        }
        InternalToolProxy toolProxy = labelToTool.get(label);
        if (!isToolAccessibleBy(toolProxy, accessLevels, operatorClientId)) {
            String errorMessage = toolProxy.getRequiredFeature() == null ?
                    String.format("Tool %s is only accessible by %s", label, toolProxy.getAllowedRoles())
                    : String.format("Tool %s is only accessible by %s and feature %s", label,
                            toolProxy.getAllowedRoles(), toolProxy.getRequiredFeature());

            throw new InternalToolAccessDeniedException(errorMessage);
        }
        return toolProxy;
    }

    /**
     * Получить список всех категорий, в которых есть доступные клиенту отчеты. Каждая категория будет содержать список
     * отчетов из нее, к которым у пользователя есть доступ
     *
     * @param accessLevels уровни доступа запрашивающего список пользователя
     */
    public Map<InternalToolCategory, List<InternalToolProxy>> getInternalToolsByCategory(
            Collection<InternalToolAccessRole> accessLevels, ClientId operatorClientId) {
        Map<InternalToolCategory, List<InternalToolProxy>> result = new EnumMap<>(InternalToolCategory.class);
        for (InternalToolProxy<?> proxy : labelToTool.values()) {
            if (!isToolAccessibleBy(proxy, accessLevels, operatorClientId)) {
                continue;
            }
            if (!result.containsKey(proxy.getCategory())) {
                result.put(proxy.getCategory(), new ArrayList<>());
            }
            result.get(proxy.getCategory()).add(proxy);
        }
        return result;
    }

    /**
     * Проверить, можно ли получить указанный прокси с заданным набором прав
     */
    boolean isToolAccessibleBy(InternalToolProxy proxy, Collection<InternalToolAccessRole> effectiveRoles,
                               ClientId operatorClientId) {
        return isToolAccessibleByAccessRoles(proxy, effectiveRoles)
                && isToolAccessibleByRequiredFeature(proxy, operatorClientId);
    }

    private boolean isToolAccessibleByAccessRoles(InternalToolProxy proxy, Collection<InternalToolAccessRole> effectiveRoles) {
        Set<InternalToolAccessRole> internalToolAccessRoles = EnumSet.copyOf(proxy.getAllowedRoles());
        return internalToolAccessRoles.removeAll(effectiveRoles);
    }

    private boolean isToolAccessibleByRequiredFeature(InternalToolProxy proxy, ClientId operatorClientId) {
        FeatureName requiredFeature = proxy.getRequiredFeature();

        if (requiredFeature == null) {
            return true;
        }

        return featureService.isEnabledForClientId(operatorClientId, requiredFeature);
    }
}
