package ru.yandex.direct.internaltools.core.bootstrap;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.primitives.Ints;
import org.apache.commons.lang3.ArrayUtils;

import ru.yandex.direct.internaltools.core.annotations.input.CheckBox;
import ru.yandex.direct.internaltools.core.annotations.input.Date;
import ru.yandex.direct.internaltools.core.annotations.input.DateTime;
import ru.yandex.direct.internaltools.core.annotations.input.File;
import ru.yandex.direct.internaltools.core.annotations.input.Group;
import ru.yandex.direct.internaltools.core.annotations.input.Hidden;
import ru.yandex.direct.internaltools.core.annotations.input.Input;
import ru.yandex.direct.internaltools.core.annotations.input.MultipleSelect;
import ru.yandex.direct.internaltools.core.annotations.input.Number;
import ru.yandex.direct.internaltools.core.annotations.input.NumericId;
import ru.yandex.direct.internaltools.core.annotations.input.Select;
import ru.yandex.direct.internaltools.core.annotations.input.ShardSelect;
import ru.yandex.direct.internaltools.core.annotations.input.Text;
import ru.yandex.direct.internaltools.core.annotations.input.TextArea;
import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.exception.InternalToolInitialisationException;
import ru.yandex.direct.internaltools.core.input.InternalToolInput;
import ru.yandex.direct.internaltools.core.input.InternalToolInputGroup;
import ru.yandex.direct.internaltools.core.input.InternalToolInputPreProcessor;
import ru.yandex.direct.internaltools.core.input.InternalToolInputType;
import ru.yandex.direct.internaltools.core.input.processors.CurrentDatePreProcessor;
import ru.yandex.direct.internaltools.core.input.processors.CurrentDateTimePreProcessor;
import ru.yandex.direct.internaltools.core.util.FieldUtil;
import ru.yandex.direct.validation.builder.Constraint;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.internaltools.core.util.FieldUtil.getAccessor;
import static ru.yandex.direct.internaltools.core.util.FieldUtil.getExtractor;
import static ru.yandex.direct.internaltools.core.util.FieldUtil.getFieldName;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;

public class InternalToolInputBootstrap {
    public static final String FIELD_LEN_ARG = "field_len";
    public static final String MIN_NUM_VALUE = "min_value";
    public static final String MAX_NUM_VALUE = "max_value";
    public static final String MAX_LEN_ARG = "max_value_len";
    public static final String COLUMNS_ARG = "columns";
    public static final String ROWS_ARG = "rows";
    public static final String DEFAULT_GROUP_NAME = "";
    public static final Integer DEFAULT_GROUP_PRIORITY = 0;
    private static final Integer MIN_SHARD = 1;

    private InternalToolInputBootstrap() {
    }

    /**
     * Получить заполненный и отсортированный список групп ввода из сигнатуры класса ввода
     */
    public static <T extends InternalToolParameter> List<InternalToolInputGroup<T>> groupsFromParamsClass(
            Class<T> inputClass, List<InternalToolInputPreProcessor<?>> preProcessors, int shardNum) {
        Map<InternalToolInputGroup<T>, List<InternalToolInput<T, ?>>> map = new HashMap<>();
        for (Field field : inputClass.getDeclaredFields()) {
            InternalToolInput<T, ?> input = inputFromField(inputClass, field, preProcessors, shardNum);
            if (input == null) {
                continue;
            }
            InternalToolInputGroup<T> group = groupFromField(field);
            List<InternalToolInput<T, ?>> inputList = map.computeIfAbsent(group, a -> new ArrayList<>());
            inputList.add(input);
        }

        return map.entrySet().stream()
                .sorted((a, b) -> Ints.compare(b.getKey().getPriority(), a.getKey().getPriority()))
                .map(e -> new InternalToolInputGroup<>(e.getKey().getName(), e.getKey().getPriority(), e.getValue()))
                .collect(toList());
    }

    /**
     * Получить описание группы из описания поля
     */
    public static <T extends InternalToolParameter> InternalToolInputGroup<T> groupFromField(Field field) {
        Group group = field.getAnnotation(Group.class);
        if (group == null) {
            return new InternalToolInputGroup<>(DEFAULT_GROUP_NAME, DEFAULT_GROUP_PRIORITY);
        }
        return new InternalToolInputGroup<>(group.name(), group.priority());
    }

    /**
     * Построить описание поля ввода по переданным параметрам
     *
     * @param inputClass класс, который представляет в виде POJO все входные параметры, включая поле, описание которого
     *                   надо сгенерировать
     * @param field      поле описание которого  надо сгенерировать
     * @param shardNum   количество шардов в директе (нужно для описания поля ShardSelect)
     */
    public static <T extends InternalToolParameter> InternalToolInput<T, ?> inputFromField(Class<T> inputClass,
                                                                                           Field field, List<InternalToolInputPreProcessor<?>> preProcessors, int shardNum) {
        Input inputData = field.getAnnotation(Input.class);
        if (inputData == null) {
            return null;
        }
        checkClass(inputClass, field);

        Class<?> cls = field.getType();
        if (field.isAnnotationPresent(Hidden.class)) {
            InternalToolInput.Builder<T, Object> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);
            return hiddenInput(builder, field.getAnnotation(Hidden.class));
        } else if ((Integer.class.isAssignableFrom(cls) || int.class.isAssignableFrom(cls)) && field
                .isAnnotationPresent(ShardSelect.class)) {
            InternalToolInput.Builder<T, Integer> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);
            List<Integer> shards = IntStream.rangeClosed(MIN_SHARD, shardNum).boxed().collect(Collectors.toList());
            return selectInput(builder, shards, MIN_SHARD);
        } else if (Set.class.isAssignableFrom(cls) && isValidGenericTypeForMultipleSelect(field)
                && field.isAnnotationPresent(MultipleSelect.class)) {
            Class<?> genericType = FieldUtil.getGenericType(field);
            InternalToolInput.Builder<T, Object> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);

            //noinspection ConstantConditions - genericType точно не null т.к. есть проверка в методе isValidGenericTypeForMultipleSelect
            if (genericType.isEnum()) {
                return enumMultipleSelectInput(builder, genericType, field.getAnnotation(MultipleSelect.class));
            } else if (Long.class.isAssignableFrom(genericType)) {
                return longMultipleSelectInput(builder, field.getAnnotation(MultipleSelect.class));
            } else {
                return stringMultipleSelectInput(builder, field.getAnnotation(MultipleSelect.class));
            }
        } else if (Boolean.class.isAssignableFrom(cls) || boolean.class.isAssignableFrom(cls)) {
            InternalToolInput.Builder<T, Boolean> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);
            return checkBox(builder, field.getAnnotation(CheckBox.class));
        } else if (Long.class.isAssignableFrom(cls) || long.class.isAssignableFrom(cls)) {
            InternalToolInput.Builder<T, Long> builder = getFilledBuilder(inputClass, field, preProcessors, inputData);
            return numberInput(builder, field.getAnnotation(Number.class), field.getAnnotation(NumericId.class));
        } else if (String.class.isAssignableFrom(cls)) {
            InternalToolInput.Builder<T, String> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);
            if (inputData.required()) {
                builder.addValidator(notEmpty());
            }
            return stringInput(builder, field);
        } else if (byte[].class.isAssignableFrom(cls) && field.isAnnotationPresent(File.class)) {
            InternalToolInput.Builder<T, byte[]> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);
            return fileInput(builder);
        } else if (LocalDate.class.isAssignableFrom(cls)) {
            InternalToolInput.Builder<T, LocalDate> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);
            return dateInput(builder, field.getAnnotation(Date.class));
        } else if (LocalDateTime.class.isAssignableFrom(cls)) {
            InternalToolInput.Builder<T, LocalDateTime> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);
            return dateTimeInput(builder, field.getAnnotation(DateTime.class));
        } else if (cls.isEnum()) {
            InternalToolInput.Builder<T, Object> builder =
                    getFilledBuilder(inputClass, field, preProcessors, inputData);
            return enumSelectInput(builder, cls);
        }

        throw new InternalToolInitialisationException(
                String.format("%s field has invalid type or annotations", field.getName()));
    }

    private static boolean isValidGenericTypeForMultipleSelect(Field field) {
        Class<?> clazz = FieldUtil.getGenericType(field);
        return clazz != null
                && (clazz.isEnum() || String.class.isAssignableFrom(clazz) || Long.class.isAssignableFrom(clazz));
    }

    private static <T extends InternalToolParameter, D> InternalToolInput.Builder<T, D> getFilledBuilder(
            Class<T> inputClass, Field field, List<InternalToolInputPreProcessor<?>> preProcessors,
            Input inputData) {
        //noinspection unchecked
        Class<D> fieldCls = (Class<D>) field.getType();
        InternalToolInput.Builder<T, D> builder = InternalToolInput.<T, D>builder()
                .withName(getFieldName(inputClass, field))
                .withLabel(inputData.label())
                .withDescription(inputData.description())
                .withExtractor(getExtractor(inputClass, field, fieldCls))
                .withRequired(inputData.required());

        for (InternalToolInputPreProcessor<?> preProcessor : preProcessors) {
            for (Class<? extends InternalToolInputPreProcessor> cls : inputData.processors()) {
                if (cls.equals(preProcessor.getClass())) {
                    //noinspection unchecked
                    builder.addProcessor((InternalToolInputPreProcessor<D>) preProcessor);
                    break;
                }
            }
        }

        return builder;
    }

    private static void checkClass(Class<?> inputClass, Field field) {
        if (Modifier.isPublic(field.getModifiers())) {
            return;
        }
        if (getAccessor(inputClass, field) == null) {
            throw new InternalToolInitialisationException(
                    String.format("Should have getter for field %s::%s", inputClass.getSimpleName(), field.getName()));
        }
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, Object> hiddenInput(
            InternalToolInput.Builder<T, Object> builder, Hidden hiddenData) {
        if (!"".equals(hiddenData.defaultValue())) {
            builder.withDefaultValue(hiddenData.defaultValue());
        }
        return builder
                .withInputType(InternalToolInputType.HIDDEN)
                .build();
    }

    private static <T extends InternalToolParameter, D> InternalToolInput<T, D> selectInput(
            InternalToolInput.Builder<T, D> builder,
            List<D> choices, @Nullable D defaultValue) {
        if (defaultValue != null) {
            builder.withDefaultValue(defaultValue);
        }
        if (!choices.isEmpty()) {
            builder.withAllowedValues(choices)
                    .addValidator(inSet(ImmutableSet.<D>builder().addAll(choices).build()));
        }
        return builder
                .withInputType(InternalToolInputType.SELECT)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, Object> enumSelectInput(
            InternalToolInput.Builder<T, Object> builder,
            Class<?> cls) {
        if (!cls.isEnum()) {
            throw new InternalToolInitialisationException("Method supports only enums");
        }

        List<Object> choices = Arrays.asList(cls.getEnumConstants());
        if (choices.isEmpty()) {
            throw new InternalToolInitialisationException("Empty enums not supported");
        }
        return selectInput(builder, choices, choices.get(0));
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, String> stringSelectInput(
            InternalToolInput.Builder<T, String> builder, Select selectData) {
        List<String> choices = Arrays.asList(selectData.choices());
        if (choices.isEmpty() && !selectData.preprocessed()) {
            throw new InternalToolInitialisationException("Empty selects not supported");
        }
        String defaultValue = Iterables.getFirst(choices, null);
        if (!"".equals(selectData.defaultValue())) {
            defaultValue = selectData.defaultValue();
        }
        return selectInput(builder, choices, defaultValue);
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, Object> multipleSelectInput(
            InternalToolInput.Builder<T, Object> builder,
            Object[] choices,
            String[] defaultValues) {
        if (!ArrayUtils.isEmpty(defaultValues)) {
            builder.withDefaultValue(defaultValues);
        }

        if (!ArrayUtils.isEmpty(choices)) {
            List<Object> allowedValues = Arrays.asList(choices);
            Constraint validator = eachInSet(ImmutableSet.builder().addAll(allowedValues).build());
            //noinspection unchecked
            builder.withAllowedValues(allowedValues)
                    .addValidator(validator);
        }

        return builder
                .withInputType(InternalToolInputType.MULTIPLE_SELECT)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, Object> enumMultipleSelectInput(
            InternalToolInput.Builder<T, Object> builder, Class<?> cls,
            MultipleSelect selectData) {
        if (!cls.isEnum()) {
            throw new InternalToolInitialisationException("Method supports only enums");
        }

        if (ArrayUtils.isEmpty(cls.getEnumConstants()) && !selectData.preprocessed()) {
            throw new InternalToolInitialisationException("Empty enums not supported");
        }

        if (!ArrayUtils.isEmpty(selectData.choices())) {
            throw new InternalToolInitialisationException("Choices not supported for enum MultipleSelect");
        }

        return multipleSelectInput(builder, cls.getEnumConstants(), selectData.defaultValues());
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, Object> stringMultipleSelectInput(
            InternalToolInput.Builder<T, Object> builder,
            MultipleSelect selectData) {
        if (ArrayUtils.isEmpty(selectData.choices()) && !selectData.preprocessed()) {
            throw new InternalToolInitialisationException("Empty selects not supported");
        }

        return multipleSelectInput(builder, selectData.choices(), selectData.defaultValues());
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, Object> longMultipleSelectInput(
            InternalToolInput.Builder<T, Object> builder,
            MultipleSelect selectData) {
        if (ArrayUtils.isEmpty(selectData.choices()) && !selectData.preprocessed()) {
            throw new InternalToolInitialisationException("Empty selects not supported");
        }

        try {
            Long[] choices = Arrays.stream(selectData.choices())
                    .map(Long::valueOf)
                    .toArray(Long[]::new);

            return multipleSelectInput(builder, choices, selectData.defaultValues());
        } catch (NumberFormatException e) {
            throw new InternalToolInitialisationException("Invalid type of choices for long multiSelect", e);
        }
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, Boolean> checkBox(
            InternalToolInput.Builder<T, Boolean> builder,
            @Nullable CheckBox cb) {
        if (cb != null && cb.checked()) {
            builder.withDefaultValue(true);
        }
        return builder
                .withInputType(InternalToolInputType.CHECKBOX)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, Long> numberInput(
            InternalToolInput.Builder<T, Long> builder,
            @Nullable Number nm, @Nullable NumericId nid) {
        if (nm != null) {
            builder.withDefaultValue(nm.defaultValue());
            long min = nm.minValue();
            long max = nm.maxValue();
            if (min != Long.MIN_VALUE || max != Long.MAX_VALUE) {
                if (min >= max) {
                    throw new InternalToolInitialisationException("Minimal number value must be less than maximal");
                }

                Map<String, Object> args = new HashMap<>();
                if (min != Long.MIN_VALUE) {
                    builder.addValidator(notLessThan(min));
                    args.put(MIN_NUM_VALUE, min);
                }
                if (max != Long.MAX_VALUE) {
                    builder.addValidator(notGreaterThan(max));
                    args.put(MAX_NUM_VALUE, max);
                }
                builder.withArgs(args);
            }
        } else if (nid != null) {
            if (nid.defaultValue() != Long.MIN_VALUE) {
                builder.withDefaultValue(nid.defaultValue());
            }
            long min = 1;
            if (nid.canBeZero()) {
                min = 0;
            }
            builder.addValidator(notLessThan(min));
            builder.withArgs(Collections.singletonMap(MIN_NUM_VALUE, min));
        }
        return builder
                .withInputType(InternalToolInputType.NUMBER)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, String> textInput(
            InternalToolInput.Builder<T, String> builder,
            @Nullable Text tf) {
        if (tf != null) {
            if (!"".equals(tf.defaultValue())) {
                builder.withDefaultValue(tf.defaultValue());
            }

            Map<String, Object> args = new HashMap<>();
            if (tf.fieldLen() > 0) {
                args.put(FIELD_LEN_ARG, tf.fieldLen());
            }
            if (tf.valueMaxLen() > 0) {
                args.put(MAX_LEN_ARG, tf.valueMaxLen());
                builder.addValidator(maxStringLength(tf.valueMaxLen()));
            }
            builder.withArgs(args);
        }
        return builder
                .withInputType(InternalToolInputType.TEXT)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, String> textArea(
            InternalToolInput.Builder<T, String> builder,
            TextArea ta) {
        if (!"".equals(ta.defaultValue())) {
            builder.withDefaultValue(ta.defaultValue());
        }

        Map<String, Object> args = new HashMap<>();
        if (ta.columns() > 0) {
            args.put(COLUMNS_ARG, ta.columns());
        }
        if (ta.rows() > 0) {
            args.put(ROWS_ARG, ta.rows());
        }
        return builder
                .withInputType(InternalToolInputType.TEXTAREA)
                .withArgs(args)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, LocalDate> dateInput(
            InternalToolInput.Builder<T, LocalDate> builder, @Nullable Date date) {
        if (date != null) {
            if (date.today() && !"".equals(date.defaultValue())) {
                throw new InternalToolInitialisationException(
                        String.format("Date annotation for %s field has both today and defaultValue marker",
                                builder.getName()));
            } else if (date.today()) {
                builder.addProcessor(CurrentDatePreProcessor.DEFAULT);
            } else if (!"".equals(date.defaultValue())) {
                try {
                    builder.withDefaultValue(LocalDate.parse(date.defaultValue()));
                } catch (Exception e) {
                    throw new InternalToolInitialisationException(
                            "Date annotation for %s field has unparsable default value", e);
                }
            }
        }
        return builder
                .withInputType(InternalToolInputType.DATE)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, LocalDateTime> dateTimeInput(
            InternalToolInput.Builder<T, LocalDateTime> builder, @Nullable DateTime dateTime) {
        if (dateTime != null) {
            if (dateTime.now() && !"".equals(dateTime.defaultValue())) {
                throw new InternalToolInitialisationException(
                        String.format("DateTime annotation for %s field has both now and defaultValue marker",
                                builder.getName()));
            } else if (dateTime.now()) {
                if (dateTime.dayShift() == 0) {
                    builder.addProcessor(CurrentDateTimePreProcessor.DEFAULT);
                } else {
                    int shift = dateTime.dayShift() > 0 ?
                            dateTime.dayShift() :
                            dateTime.dayShift() + 1;
                    builder.addProcessor(new CurrentDateTimePreProcessor(time -> time
                            .toLocalDate()
                            .atTime(0, 0)
                            .plusDays(shift)
                    ));
                }
            } else if (!"".equals(dateTime.defaultValue())) {
                try {
                    builder.withDefaultValue(LocalDateTime.parse(dateTime.defaultValue()));
                } catch (Exception e) {
                    throw new InternalToolInitialisationException(
                            "DateTime annotation for %s field has unparsable default value", e);
                }
            }
        }
        return builder
                .withInputType(InternalToolInputType.DATETIME)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, byte[]> fileInput(
            InternalToolInput.Builder<T, byte[]> builder) {
        return builder
                .withInputType(InternalToolInputType.FILE)
                .build();
    }

    private static <T extends InternalToolParameter> InternalToolInput<T, String> stringInput(
            InternalToolInput.Builder<T, String> builder, Field field) {
        if (field.isAnnotationPresent(Select.class)) {
            return stringSelectInput(builder, field.getAnnotation(Select.class));
        } else if (field.isAnnotationPresent(TextArea.class)) {
            return textArea(builder, field.getAnnotation(TextArea.class));
        }
        return textInput(builder, field.getAnnotation(Text.class));
    }

}
