package ru.yandex.direct.internaltools.core.bootstrap;

import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.InternalToolProxy;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Disclaimers;
import ru.yandex.direct.internaltools.core.annotations.tool.RequiredFeature;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.enrich.InternalToolEnrichProcessorFactory;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.exception.InternalToolInitialisationException;
import ru.yandex.direct.internaltools.core.input.InternalToolInput;
import ru.yandex.direct.internaltools.core.input.InternalToolInputGroup;
import ru.yandex.direct.internaltools.core.input.InternalToolInputPreProcessor;
import ru.yandex.direct.internaltools.core.input.InternalToolInputType;

import static ru.yandex.direct.internaltools.core.bootstrap.InternalToolInputBootstrap.groupsFromParamsClass;

public class InternalToolProxyBootstrap {
    private static final Pattern LABEL_PATTERN = Pattern.compile("[a-z0-9_]+");

    private InternalToolProxyBootstrap() {
    }

    /**
     * Построить прокси-объект над инструментом из класса инструмента
     */
    static <T extends InternalToolParameter> InternalToolProxy<T> proxyFromTools(BaseInternalTool<T> internalTool,
                                                                                 InternalToolEnrichProcessorFactory enrichProcessorFactory,
                                                                                 List<InternalToolInputPreProcessor<?>> preProcessors, int shardNum) {
        Tool tool = extractToolDescription(internalTool);
        String label = tool.label();
        if (!LABEL_PATTERN.matcher(label).matches()) {
            throw new InternalToolInitialisationException(String.format("Internal tool %s has invalid label %s",
                    internalTool.getClass().getCanonicalName(), label));
        }
        //noinspection unchecked
        Class<T> inputClass = (Class<T>) tool.consumes();
        InternalToolProxy.Builder<T> builder = new InternalToolProxy.Builder<T>()
                .withInternalTool(internalTool)
                .withLabel(label)
                .withAllowedRoles(extractToolAccessRoles(internalTool))
                .withRequiredFeature(extractToolRequiredFeature(internalTool))
                .withCategory(extractToolCategory(internalTool))
                .withName(tool.name())
                .withDescription(tool.description())
                .withAction(extractAction(internalTool))
                .withType(tool.type())
                .withDisclaimers(extractDisclaimers(internalTool))
                .withInputClass(inputClass)
                .withInputGroups(groupsFromParamsClass(inputClass, preProcessors, shardNum))
                .withEnrichProcessorFactory(enrichProcessorFactory);
        return internalTool.preCreate(builder).build();
    }

    private static Tool extractToolDescription(BaseInternalTool<?> internalTool) {
        Tool tool = internalTool.getClass().getAnnotation(Tool.class);
        if (tool == null) {
            throw new InternalToolInitialisationException(String.format("Internal tool %s has no @Tool annotation",
                    internalTool.getClass().getCanonicalName()));
        }
        return tool;
    }

    private static Set<InternalToolAccessRole> extractToolAccessRoles(BaseInternalTool<?> internalTool) {
        Set<InternalToolAccessRole> roles = EnumSet.noneOf(InternalToolAccessRole.class);

        AccessGroup group = internalTool.getClass().getAnnotation(AccessGroup.class);
        if (group == null || group.value().length == 0) {
            roles.add(InternalToolAccessRole.SUPER);
            roles.add(InternalToolAccessRole.SUPERREADER);
        } else {
            roles.addAll(Arrays.asList(group.value()));
        }
        expandRoles(roles);
        return roles;
    }

    private static FeatureName extractToolRequiredFeature(BaseInternalTool<?> internalTool) {
        RequiredFeature requiredFeature = internalTool.getClass().getAnnotation(RequiredFeature.class);
        return requiredFeature == null ? null : requiredFeature.value();
    }

    public static <T extends InternalToolParameter> Set<String> extractFileFieldsNames(
            List<InternalToolInputGroup<T>> inputGroups) {
        return inputGroups.stream()
                .map(InternalToolInputGroup::getInputListWithoutPreprocessing)
                .flatMap(List::stream)
                .filter(i -> InternalToolInputType.FILE == i.getInputType())
                .map(InternalToolInput::getName)
                .collect(Collectors.toSet());
    }

    static void expandRoles(Set<InternalToolAccessRole> roles) {
        boolean rolesChanged;
        do {
            rolesChanged = false;
            for (InternalToolAccessRole role : roles) {
                rolesChanged = roles.addAll(role.getParentRoles()) || rolesChanged;
            }
        } while (rolesChanged);
    }

    private static InternalToolCategory extractToolCategory(BaseInternalTool<?> internalTool) {
        Category category = internalTool.getClass().getAnnotation(Category.class);
        return category == null ? InternalToolCategory.OTHER : category.value();
    }

    private static List<String> extractDisclaimers(BaseInternalTool<?> internalTool) {
        Disclaimers disclaimers = internalTool.getClass().getAnnotation(Disclaimers.class);
        return disclaimers == null ? Collections.emptyList() : Arrays.asList(disclaimers.value());
    }

    private static InternalToolAction extractAction(BaseInternalTool<?> internalTool) {
        Action action = internalTool.getClass().getAnnotation(Action.class);
        return action == null ? InternalToolAction.SHOW : action.value();
    }
}
