package ru.yandex.direct.internaltools.core.input;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.exception.InternalToolInitialisationException;
import ru.yandex.direct.internaltools.core.util.InternalToolFieldExtractor;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

/**
 * Описание поля ввода, в сериализованном виде достаточное для построения этого поля пользовательским интерфейсом.
 * Также внутри этого класса осуществляется валидация значений полей ввода этого типа
 */
@ParametersAreNonnullByDefault
public class InternalToolInput<T extends InternalToolParameter, D> {
    public static class Builder<T extends InternalToolParameter, D> {
        private InternalToolInputType inputType;
        private String label;
        private String description;
        private String name;
        private List<D> allowedValues;
        private Map<String, Object> args = Collections.emptyMap();
        private D defaultValue = null;
        private boolean required;
        private InternalToolFieldExtractor<T, D> extractor;
        private List<Constraint<D, Defect>> validators = new ArrayList<>();
        private List<InternalToolInputPreProcessor<D>> processors = new ArrayList<>();

        public Builder<T, D> withInputType(InternalToolInputType inputType) {
            this.inputType = inputType;
            return this;
        }

        public Builder<T, D> withLabel(String label) {
            this.label = label;
            return this;
        }

        public String getDescription() {
            return description;
        }

        public Builder<T, D> withDescription(String description) {
            this.description = description;
            return this;
        }

        public String getName() {
            return name;
        }

        public Builder<T, D> withName(String name) {
            this.name = name;
            return this;
        }

        public Builder<T, D> withAllowedValues(@Nullable List<D> allowedValues) {
            this.allowedValues = allowedValues;
            return this;
        }

        public Builder<T, D> withArgs(Map<String, Object> args) {
            this.args = args;
            return this;
        }

        public Builder<T, D> withDefaultValue(@Nullable D defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        public Builder<T, D> withRequired(boolean required) {
            this.required = required;
            return this;
        }

        public Builder<T, D> withExtractor(InternalToolFieldExtractor<T, D> extractor) {
            this.extractor = extractor;
            return this;
        }

        public Builder<T, D> withProcessors(List<InternalToolInputPreProcessor<D>> processors) {
            this.processors = processors;
            return this;
        }

        public Builder<T, D> addProcessor(InternalToolInputPreProcessor<D> processor) {
            processors.add(processor);
            return this;
        }

        public Builder<T, D> withValidators(List<Constraint<D, Defect>> validators) {
            this.validators = validators;
            return this;
        }

        public Builder<T, D> addValidator(Constraint<D, Defect> validator) {
            validators.add(validator);
            return this;
        }

        public InternalToolInput<T, D> build() {
            InternalToolInput.Builder<T, D> builder = this;
            if (processors != null && !processors.isEmpty()) {
                for (InternalToolInputPreProcessor<D> processor : processors) {
                    builder = processor.preCreate(builder);
                }
            }
            if (builder.name == null || builder.inputType == null || builder.label == null
                    || builder.description == null || builder.args == null || builder.extractor == null
                    || builder.validators == null || builder.processors == null) {
                throw new InternalToolInitialisationException("Internal tool input params cannot be null");
            }
            if (!FIRED_NAME_PATTERN.matcher(builder.name).matches()) {
                throw new InternalToolInitialisationException("Name not present or not correct");
            }
            if (builder.allowedValues != null && builder.validators.isEmpty()) {
                throw new InternalToolInitialisationException("When allowed values are provided, you need also "
                        + "provide validation");
            }
            return new InternalToolInput<>(builder.inputType, builder.label, builder.name, builder.description,
                    builder.args, builder.allowedValues, builder.defaultValue, builder.required, builder.extractor,
                    builder.processors, builder.validators);
        }
    }

    private static final Pattern FIRED_NAME_PATTERN = Pattern.compile("[A-Za-z0-9_]+");

    private final InternalToolInputType inputType;
    private final String label;
    private final String description;
    private final String name;
    private final Map<String, Object> args;
    private final List<D> allowedValues;
    private final D defaultValue;
    private final boolean required;
    private final InternalToolFieldExtractor<T, D> extractor;
    private final List<Constraint<D, Defect>> validators;
    private final List<InternalToolInputPreProcessor<D>> preProcessors;

    public InternalToolInput(InternalToolInputType inputType, String label, String name, String description,
                             Map<String, Object> args, @Nullable List<D> allowedValues, @Nullable D defaultValue, boolean required,
                             InternalToolFieldExtractor<T, D> extractor, List<InternalToolInputPreProcessor<D>> preProcessors,
                             List<Constraint<D, Defect>> validators) {
        this.inputType = inputType;
        this.label = label;
        this.name = name;
        this.description = description;
        this.args = args;
        this.allowedValues = allowedValues;
        this.defaultValue = defaultValue;
        this.required = required;
        this.extractor = extractor;
        this.validators = new ArrayList<>(validators);
        this.preProcessors = preProcessors;
    }

    public static <T extends InternalToolParameter, D> Builder<T, D> builder() {
        return new Builder<>();
    }

    /**
     * Добавить к переданному валидатору проверки на текущее поле, если это нужно
     *
     * @param param Валидируемый объект
     */
    public void addValidation(ItemValidationBuilder<T, Defect> validationBuilder, T param) {
        InternalToolInput<T, D> input = applyPreProcessors(false);
        if (!input.required && input.validators.isEmpty()) {
            return;
        }
        D value = input.extractor.getValue(param);
        ItemValidationBuilder<D, Defect> itemValidationBuilder =
                validationBuilder.item(value, name);
        if (input.required) {
            itemValidationBuilder.check(notNull());
        }
        for (Constraint<D, Defect> validator : input.validators) {
            itemValidationBuilder.check(validator, When.notNull());
        }
    }

    public InternalToolInputType getInputType() {
        return inputType;
    }

    public String getLabel() {
        return label;
    }

    public String getName() {
        return name;
    }

    public Map<String, Object> getArgs() {
        return args;
    }

    public List<D> getAllowedValues() {
        return allowedValues;
    }

    public D getDefaultValue() {
        return defaultValue;
    }

    public boolean isRequired() {
        return required;
    }

    public String getDescription() {
        return description;
    }

    private InternalToolInput.Builder<T, D> getFilledBuilder() {
        return new InternalToolInput.Builder<T, D>()
                .withLabel(label)
                .withName(name)
                .withDescription(description)
                .withArgs(new HashMap<>(args))
                .withInputType(inputType)
                .withExtractor(extractor)
                .withRequired(required)
                .withAllowedValues(allowedValues)
                .withDefaultValue(defaultValue)
                .withValidators(new ArrayList<>(validators));
    }

    public InternalToolInput<T, D> applyPreProcessors() {
        return applyPreProcessors(true);
    }

    public InternalToolInput<T, D> applyPreProcessors(boolean forSend) {
        if (preProcessors.isEmpty()) {
            return this;
        }
        InternalToolInput.Builder<T, D> builder = getFilledBuilder();
        for (InternalToolInputPreProcessor<D> processor : preProcessors) {
            if (forSend) {
                builder = processor.preSend(builder);
            } else {
                builder = processor.preReceive(builder);
            }
        }
        return builder.build();
    }

    public List<InternalToolInputPreProcessor<D>> getPreProcessors() {
        return preProcessors;
    }

    public List<Constraint<D, Defect>> getValidators() {
        return validators;
    }
}
