package ru.yandex.direct.internaltools.core.util;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.internaltools.core.exception.InternalToolProcessingException;

import static com.fasterxml.jackson.annotation.JsonProperty.USE_DEFAULT_NAME;

@ParametersAreNonnullByDefault
public class FieldUtil {
    private FieldUtil() {
    }

    /**
     * Получить имя поля в сериализованном представлении класса с параметрами (т.е. с учетом аннотации JsonProperty)
     */
    public static String getFieldName(Class<?> inputClass, Field field) {
        JsonProperty jsonProperty = field.getAnnotation(JsonProperty.class);
        if (jsonProperty == null) {
            Method getter = getAccessor(inputClass, field);
            if (getter != null) {
                jsonProperty = getter.getAnnotation(JsonProperty.class);
            }
        }
        if (jsonProperty != null && !USE_DEFAULT_NAME.equals(jsonProperty.value())) {
            return jsonProperty.value();
        }
        return field.getName();
    }

    /**
     * Получить метод переданного класса, который потенциально является геттером для переданного поля. Если такого нет,
     * вернуть null
     */
    public static Method getAccessor(Class<?> inputClass, Field field) {
        String name = field.getName();
        String capName = name.substring(0, 1).toUpperCase() + name.substring(1);

        String prefix = "get";
        if (boolean.class.isAssignableFrom(field.getType())) {
            prefix = "is";
        }
        Method method;
        try {
            method = inputClass.getMethod(prefix + capName);
        } catch (NoSuchMethodException e) {
            return null;
        }
        if (field.getType() != method.getReturnType() || !Modifier.isPublic(method.getModifiers())) {
            return null;
        }
        return method;
    }

    /**
     * Получить объект, который умеет доставать переданное поле из экземпляров указанного класса
     */
    public static <T, D> InternalToolFieldExtractor<T, D> getExtractor(Class<T> inputClass, Field field,
                                                                       Class<D> fieldClass) {
        if (Modifier.isPublic(field.getModifiers())) {
            return o -> {
                try {
                    return convert(field.get(o), fieldClass);
                } catch (IllegalAccessException e) {
                    return null;
                }
            };
        }
        Method method = getAccessor(inputClass, field);
        if (method != null) {
            return o -> {
                try {
                    return convert(method.invoke(o), fieldClass);
                } catch (IllegalAccessException | InvocationTargetException e) {
                    return null;
                }
            };
        }

        throw new InternalToolProcessingException(
                String.format("Cannot get extractor for %s field in class %s", field.getName(), inputClass));
    }

    private static <D> D convert(Object o, Class<D> cls) {
        if (!cls.isPrimitive()) {
            return cls.cast(o);
        }
        //noinspection unchecked
        return (D) o;
    }

    /**
     * Получить класс, которым параметризованно generic-поле
     *
     * @return класс, которым параметризованно generic-поле или null если поле не параметризованно
     */
    @Nullable
    public static Class<?> getGenericType(Field field) {
        Type type = field.getGenericType();
        if (type instanceof ParameterizedType) {
            ParameterizedType parameterizedType = (ParameterizedType) type;
            return (Class<?>) parameterizedType.getActualTypeArguments()[0];
        }

        return null;
    }
}
