package ru.yandex.direct.internaltools.configuration;

import java.util.Collection;

import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import org.springframework.web.context.annotation.RequestScope;

import ru.yandex.direct.common.configuration.UacYdbConfiguration;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.communication.config.CommunicationConfiguration;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.configuration.HourglassYdbStorageConfiguration;
import ru.yandex.direct.core.configuration.CoreConfiguration;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.uac.grut.GrutContext;
import ru.yandex.direct.core.entity.uac.grut.RequestScopeGrutContext;
import ru.yandex.direct.core.util.GrutTraceCallback;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.ess.client.EssClientConfiguration;
import ru.yandex.direct.hourglass.client.HourglassClient;
import ru.yandex.direct.hourglass.client.HourglassClientImpl;
import ru.yandex.direct.hourglass.ydb.schedulerinstances.YdbScheduleInstancesRepository;
import ru.yandex.direct.hourglass.ydb.storage.YdbStorageImpl;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.InternalToolsRegistry;
import ru.yandex.direct.internaltools.core.annotations.tool.BetaOnly;
import ru.yandex.direct.internaltools.core.annotations.tool.HideInProduction;
import ru.yandex.direct.internaltools.core.bootstrap.InternalToolEnrichProcessorFactoryBootstrap;
import ru.yandex.direct.internaltools.core.bootstrap.InternalToolsRegistryBootstrap;
import ru.yandex.direct.internaltools.core.enrich.InternalToolEnrichDataFetcher;
import ru.yandex.direct.internaltools.core.enrich.InternalToolEnrichProcessorFactory;
import ru.yandex.direct.internaltools.core.input.InternalToolInputPreProcessor;
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean;
import ru.yandex.direct.oneshot.core.configuration.OneshotCoreConfiguration;
import ru.yandex.direct.scheduler.configuration.SchedulerConfiguration;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.grut.client.GrutClient;
import ru.yandex.grut.client.GrutGrpcClient;
import ru.yandex.grut.client.ServiceHolder;

import static ru.yandex.direct.solomon.SolomonUtils.GRUT_METRICS_REGISTRY;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Configuration
@ComponentScan(
        basePackages = {
                "ru.yandex.direct.internaltools.tools",
                "ru.yandex.direct.internaltools.datafetchers"
        },
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
@Import({
        CoreConfiguration.class,
        SchedulerConfiguration.class,
        HourglassYdbStorageConfiguration.class,
        OneshotCoreConfiguration.class,
        EssClientConfiguration.class,
        UacYdbConfiguration.class,
        CommunicationConfiguration.class,
})
public class InternalToolsConfiguration {

    public static final String GRUT_CONTEXT_FOR_PROD = "prodGrutContext";
    public static final String DIRECT_CONFIG_PROD_SERVICE_HOLDER = "prod_service_holder";

    @Bean
    public InternalToolEnrichProcessorFactory enrichProcessorFactory(ApplicationContext context) {
        return InternalToolEnrichProcessorFactoryBootstrap
                .factoryFromFetchers(context.getBeansOfType(InternalToolEnrichDataFetcher.class).values());
    }

    @Bean
    public InternalToolsRegistry internalToolsRegistry(ApplicationContext context,
                                                       InternalToolEnrichProcessorFactory enrichProcessorFactory,
                                                       ShardHelper shardHelper,
                                                       FeatureService featureService) {
        Collection<BaseInternalTool> tools = context.getBeansOfType(BaseInternalTool.class).values();
        Collection<InternalToolInputPreProcessor> preProcessors
                = context.getBeansOfType(InternalToolInputPreProcessor.class).values();

        // В проде и престэйбле прячем тестовые инструменты.
        if (Environment.getCached().isProductionOrPrestable()) {
            tools = hideTestTools(tools);
        }
        // Прячем инструменты "только для бет", если это не бета
        if (!Environment.getCached().isBeta()) {
            tools = hideBetaOnlyTools(tools);
        }
        return InternalToolsRegistryBootstrap.generateRegistry(
                mapList(tools, b -> (BaseInternalTool<?>) b),
                enrichProcessorFactory,
                mapList(preProcessors, b -> (InternalToolInputPreProcessor<?>) b),
                featureService,
                shardHelper.dbShards().size()
        );
    }

    Collection<BaseInternalTool> hideTestTools(Collection<BaseInternalTool> tools) {
        return FunctionalUtils.filterList(tools,
                baseInternalTool -> !baseInternalTool.getClass().isAnnotationPresent(HideInProduction.class));
    }

    Collection<BaseInternalTool> hideBetaOnlyTools(Collection<BaseInternalTool> tools) {
        return FunctionalUtils.filterList(tools,
                baseInternalTool -> !baseInternalTool.getClass().isAnnotationPresent(BetaOnly.class));
    }

    @Bean
    public HourglassClient ydbHourglassClient(YdbStorageImpl storage,
                                              YdbScheduleInstancesRepository schedulerInstancesRepository) {
        return new HourglassClientImpl(storage, schedulerInstancesRepository);
    }

    @Bean
    @Primary
    @RequestScope
    public GrutContext grutContext(GrutClient grutClient) {
        return new RequestScopeGrutContext(grutClient);
    }

    @Bean(GRUT_CONTEXT_FOR_PROD)
    @Lazy
    @RequestScope
    public GrutContext prodGrutContext(LiveResourceFactoryBean liveResourceFactoryBean,
                                       PpcPropertiesSupport ppcPropertiesSupport,
                                       DirectConfig directConfig) {
        ServiceHolder serviceHolder = CoreConfiguration.createServiceHolder(liveResourceFactoryBean,
                directConfig.getBranch("object_api"), DIRECT_CONFIG_PROD_SERVICE_HOLDER, false);
        GrutGrpcClient grutGrpcClient =
                new GrutGrpcClient(serviceHolder, new GrutTraceCallback(ppcPropertiesSupport), GRUT_METRICS_REGISTRY);
        return new RequestScopeGrutContext(grutGrpcClient);
    }
}
