package ru.yandex.direct.internaltools.datafetchers;

import java.net.URISyntaxException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.http.client.utils.URIBuilder;

import ru.yandex.direct.internaltools.core.annotations.enrich.EnrichFetcher;
import ru.yandex.direct.internaltools.core.enrich.InternalToolEnrichDataFetcher;
import ru.yandex.direct.internaltools.core.enums.InternalToolDetailKey;

@EnrichFetcher(InternalToolDetailKey.ESS_LOGIC_OBJECTS_LINK)
@ParametersAreNonnullByDefault
public class EssLogicObjectsFetcher implements InternalToolEnrichDataFetcher<String, String> {
    private static final String LOGVIEWER_URL = "https://direct.yandex.ru/logviewer/";

    @Override
    public Map<String, String> fetch(Collection<String> topicsName) {
        return StreamEx.of(topicsName)
                .filter(topicName -> topicName != null && !topicName.isEmpty())
                .mapToEntry(this::buildUrl)
                .toMap();
    }

    public String buildUrl(String topicName) {
        String dateTo = formatDateTimeForLink(LocalDateTime.now());
        String dateFrom = formatDateTimeForLink(LocalDateTime.now().minusDays(3));
        String fragment = "ess_logic_objects,";
        fragment += dateFrom + ',';
        fragment += dateTo + ',';
        fragment += "1:101,7,binlog_time,logic_object_attr,logic_object_value,source,seqNo,gtid,reqid,";
        fragment += "topic,%" + topicName;

        URIBuilder uri = null;
        try {
            uri = new URIBuilder(LOGVIEWER_URL);
        } catch (URISyntaxException e) {
            throw new RuntimeException("Failed to build uri from " + LOGVIEWER_URL, e);
        }
        uri.setFragment(fragment);
        return uri.toString();
    }

    private String formatDateTimeForLink(LocalDateTime date) {
        return date.format(DateTimeFormatter.ofPattern("yyyyMMdd HHmmss")).replace(' ', 'T');
    }
}
