package ru.yandex.direct.internaltools.tools.additionaltargetings

import ru.yandex.direct.core.entity.additionaltargetings.repository.CampAdditionalTargetingsRepository
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool
import ru.yandex.direct.internaltools.tools.additionaltargetings.container.CampAdditionalTargetingsInfo
import ru.yandex.direct.internaltools.tools.additionaltargetings.model.CampAdditionalTargetingsAddParameters
import ru.yandex.direct.internaltools.tools.additionaltargetings.validation.AdditionalTargetingsPredicates.areCidsExists
import ru.yandex.direct.internaltools.tools.additionaltargetings.validation.AdditionalTargetingsPredicates.isJsonCorrect
import ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import java.util.function.Function

@Tool(
    name = "На кампании - просмотр, создание",
    label = "camp_additional_targetings_add_tool",
    description = "Для просмотра заполните только поле Кампании.\n" +
        "Для добавления нового таргетинга укажите список кампаний и json таргетинга для добавления.",
    consumes = CampAdditionalTargetingsAddParameters::class,
    type = InternalToolType.WRITER
)
@Category(InternalToolCategory.ADDITIONAL_TARGETINGS)
@Action(InternalToolAction.EXECUTE)
@AccessGroup(InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER)
class CampAdditionalTargetingsAddTool constructor(
    private val shardHelper: ShardHelper,
    private val campAdditionalTargetingsRepository: CampAdditionalTargetingsRepository
) : MassInternalTool<CampAdditionalTargetingsAddParameters, CampAdditionalTargetingsInfo>() {

    override fun validate(t: CampAdditionalTargetingsAddParameters) = validateObject(t) {
        property(t::campaignIds)
            .check(isStringWithValidLongIds(1), When.notNull())
            .check(
                fromPredicate(areCidsExists(shardHelper), CommonDefects.objectNotFound()),
                When.isValidAnd(When.notNull())
            )
        property(t::targeting)
            .check(fromPredicate(isJsonCorrect(), CommonDefects.invalidFormat()), When.notNull())
    }

    override fun getMassData(parameter: CampAdditionalTargetingsAddParameters): List<CampAdditionalTargetingsInfo> {
        val shardedCids = shardHelper.groupByShard(parameter.parseDistinctCampaignIds(), ShardKey.CID)
        if (!parameter.targeting.isNullOrEmpty()) {
            shardedCids.forEach { shard, chunkCids ->
                campAdditionalTargetingsRepository.insertTargetingToCampaigns(
                    shard,
                    chunkCids,
                    parameter.targeting!!,
                    parameter.comment
                )
            }
        }

        return shardedCids.stream()
            .mapKeyValue(campAdditionalTargetingsRepository::findByCids)
            .toFlatList(Function.identity())
            .map { CampAdditionalTargetingsInfo(it.cid, it.id, it.data, it.comment) }
    }
}
