package ru.yandex.direct.internaltools.tools.additionaltargetings

import ru.yandex.direct.core.entity.additionaltargetings.repository.ClientAdditionalTargetingsRepository
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool
import ru.yandex.direct.internaltools.tools.additionaltargetings.container.ClientAdditionalTargetingsInfo
import ru.yandex.direct.internaltools.tools.additionaltargetings.model.ClientAdditionalTargetingsAddParameters
import ru.yandex.direct.internaltools.tools.additionaltargetings.validation.AdditionalTargetingsPredicates.areClientIdsExists
import ru.yandex.direct.internaltools.tools.additionaltargetings.validation.AdditionalTargetingsPredicates.isJsonCorrect
import ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import java.util.function.Function

@Tool(
    name = "На клиенте - просмотр, создание",
    label = "client_additional_targetings_add_tool",
    description = "Для просмотра заполните только поле Клиенты.\n" +
        "Для добавления нового таргетинга укажите список клиентов и json таргетинга для добавления.",
    consumes = ClientAdditionalTargetingsAddParameters::class,
    type = InternalToolType.WRITER
)
@Category(InternalToolCategory.ADDITIONAL_TARGETINGS)
@Action(InternalToolAction.EXECUTE)
@AccessGroup(InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER)
class ClientAdditionalTargetingsAddTool constructor(
    private val shardHelper: ShardHelper,
    private val clientAdditionalTargetingsRepository: ClientAdditionalTargetingsRepository
) : MassInternalTool<ClientAdditionalTargetingsAddParameters, ClientAdditionalTargetingsInfo>() {

    override fun validate(t: ClientAdditionalTargetingsAddParameters) = validateObject(t) {
        property(t::clientIds)
            .check(isStringWithValidLongIds(1), When.notNull())
            .check(
                fromPredicate(areClientIdsExists(shardHelper), CommonDefects.objectNotFound()),
                When.isValidAnd(When.notNull())
            )
        property(t::targeting)
            .check(fromPredicate(isJsonCorrect(), CommonDefects.invalidFormat()), When.notNull())
    }

    override fun getMassData(parameter: ClientAdditionalTargetingsAddParameters): List<ClientAdditionalTargetingsInfo> {
        val shardedClientIds = shardHelper.groupByShard(parameter.parseDistinctClientIds(), ShardKey.CLIENT_ID)
        if (!parameter.targeting.isNullOrEmpty()) {
            shardedClientIds.forEach { shard, clientIds ->
                clientAdditionalTargetingsRepository.insertTargetingToClients(
                    shard,
                    clientIds,
                    parameter.targeting!!,
                    parameter.comment
                )
            }
        }

        return shardedClientIds.stream()
            .mapKeyValue(clientAdditionalTargetingsRepository::findByClientIds)
            .toFlatList(Function.identity())
            .map { ClientAdditionalTargetingsInfo(it.clientId, it.id, it.data, it.comment) }
    }
}
