package ru.yandex.direct.internaltools.tools.additionaltargetings

import ru.yandex.direct.core.entity.additionaltargetings.repository.ClientAdditionalTargetingsRepository
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.internaltools.core.BaseInternalTool
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.container.InternalToolResult
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.tools.additionaltargetings.model.ClientAdditionalTargetingsDeleteParameters
import ru.yandex.direct.internaltools.tools.additionaltargetings.validation.AdditionalTargetingsDeleteDefects.requiredAndPossiblyOnlyOneField
import ru.yandex.direct.internaltools.tools.additionaltargetings.validation.AdditionalTargetingsPredicates.areClientIdsExists
import ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

@Tool(
    name = "На клиенте - удаление",
    label = "client_additional_targetings_delete_tool",
    description = "Для удаления всех таргетингов клиента заполните поле Клиенты.\n" +
        "Для точечного удаления таргетингов заполните поле Таргетинги.\n" +
        "Должно быть заполнено только одно из полей.",
    consumes = ClientAdditionalTargetingsDeleteParameters::class,
    type = InternalToolType.WRITER
)
@Category(InternalToolCategory.ADDITIONAL_TARGETINGS)
@Action(InternalToolAction.DELETE)
@AccessGroup(InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER)
class ClientAdditionalTargetingsDeleteTool constructor(
    private val shardHelper: ShardHelper,
    private val clientAdditionalTargetingsRepository: ClientAdditionalTargetingsRepository
) : BaseInternalTool<ClientAdditionalTargetingsDeleteParameters> {

    override fun validate(t: ClientAdditionalTargetingsDeleteParameters) = validateObject(t) {
        check(
            fromPredicateOfNullable(
                { it.targetingIds.isNullOrBlank() xor it.clientIds.isNullOrBlank() },
                requiredAndPossiblyOnlyOneField()
            )
        )

        property(t::targetingIds)
            .check(isStringWithValidLongIds(1), When.notNull())

        property(t::clientIds)
            .check(isStringWithValidLongIds(1), When.notNull())
            .check(
                fromPredicate(areClientIdsExists(shardHelper), CommonDefects.objectNotFound()),
                When.isValidAnd(When.notNull())
            )
    }

    override fun process(t: ClientAdditionalTargetingsDeleteParameters): InternalToolResult {
        val deleted = when {
            t.targetingIds?.isNotEmpty() == true ->
                shardHelper
                    .forEachShardSequential { shard ->
                        clientAdditionalTargetingsRepository.deleteByIds(shard, t.parseDistinctTargetingIds())
                    }
                    .values.sum()
            else ->
                shardHelper
                    .groupByShard(t.parseDistinctClientIds(), ShardKey.CLIENT_ID)
                    .stream()
                    .mapKeyValue(clientAdditionalTargetingsRepository::deleteByClientIds)
                    .sum()
        }
        return InternalToolResult().withMessage("Удалено $deleted таргетингов")
    }
}
