package ru.yandex.direct.internaltools.tools.agency;

import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.agency.service.AgencyService;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToClientIdState;
import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithState;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.tools.agency.model.AgencyDealNotificationEmail;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.Result;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;

@Tool(
        name = "Email для уведомлений по сделкам (включает фичу \"Сделки\")",
        label = "agency_deal_notification_email",
        description = "Инструмент для указания адреса электронной почты для уведомлений агентству по сделкам. "
                + "Обратите внимание: при указании адреса, агентству автоматически открывается доступ к интерфейсу " +
                "сделок",
        consumes = AgencyDealNotificationEmail.class,
        type = InternalToolType.WRITER)
@Action(InternalToolAction.SET)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPPORT, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class AgencyDealNotificationEmailTool implements BaseInternalTool<AgencyDealNotificationEmail> {

    static final String AGENCY_LOGIN_NOT_FOUND = "Логин агентства не найден";
    static final String SUCCESS_MESSAGE = "Адрес электронной почты успешно изменен. Фича включена";

    private final AgencyService agencyService;
    private final RbacService rbacService;
    private final ShardHelper shardHelper;
    private final FeatureManagingService featureManagingService;

    @Autowired
    public AgencyDealNotificationEmailTool(AgencyService agencyService,
                                           RbacService rbacService, ShardHelper shardHelper,
                                           FeatureManagingService featureManagingService) {
        this.agencyService = agencyService;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
        this.featureManagingService = featureManagingService;
    }

    @Override
    public InternalToolResult process(AgencyDealNotificationEmail input) {
        String message;
        Long agencyUid = shardHelper.getUidByLogin(input.getLogin());
        long agencyClientId = shardHelper.getClientIdByLogin(input.getLogin());
        if (agencyUid == null) {
            message = AGENCY_LOGIN_NOT_FOUND;
        } else if (rbacService.getUidRole(agencyUid) != RbacRole.AGENCY) {
            message = AGENCY_LOGIN_NOT_FOUND;
        } else if ((input.getOperator().getRole() == RbacRole.MANAGER) && !rbacService
                .isOperatorManagerOfAgency(input.getOperator().getUid(), agencyUid, agencyClientId)) {
            message = AGENCY_LOGIN_NOT_FOUND;
        } else {
            ClientId agencyClient = ClientId.fromLong(agencyClientId);
            agencyService.setDealNotificationEmail(agencyClient, input.getEmail());
            Result<List<LoginClientIdChiefLoginWithState>> operationResult =
                    featureManagingService.switchFeaturesStateForClientIds(Collections.singletonList(
                            new FeatureTextIdToClientIdState()
                                    .withClientId(agencyClient)
                                    .withTextId(FeatureName.CPM_DEALS.getName())
                                    .withState(FeatureState.ENABLED)
                    ));
            if (hasValidationIssues(operationResult)) {
                throw new InternalToolValidationException("")
                        .withValidationResult(operationResult.getValidationResult());
            }
            message = SUCCESS_MESSAGE;
        }
        return new InternalToolResult().withMessage(message);
    }
}
