package ru.yandex.direct.internaltools.tools.aggregatedstatuses;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.aggregatedstatuses.model.RecalculateCampaignsStatusJobParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.common.db.PpcPropertyNames.RECALCULATE_CAMPAIGNS_STATUS_JOB_DEPTH;
import static ru.yandex.direct.common.db.PpcPropertyNames.lastProcessedCampaignId;
import static ru.yandex.direct.common.db.PpcPropertyNames.recalculateCampaignsStatusJobEnabled;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@Tool(
        name = "Управление пересчетом агрегированных статусов кампаний",
        label = "manage_recalculate_campaigns_status_job",
        description = "Контролирует таск для пересчета статусов кампаний, объявлений, групп, фраз и ретаргетингов " +
                "(глобально или на конкретном шарде)",
        consumes = RecalculateCampaignsStatusJobParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.NEW_INTERFACE)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class ManageRecalculateCampaignsStatusJobTool implements BaseInternalTool<RecalculateCampaignsStatusJobParameters> {
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ShardHelper shardHelper;

    @Autowired
    public ManageRecalculateCampaignsStatusJobTool(PpcPropertiesSupport ppcPropertiesSupport, ShardHelper shardHelper) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.shardHelper = shardHelper;
    }

    @Override
    public ValidationResult<RecalculateCampaignsStatusJobParameters, Defect> validate(
            RecalculateCampaignsStatusJobParameters jobState) {
        ItemValidationBuilder<RecalculateCampaignsStatusJobParameters, Defect> vb = ItemValidationBuilder.of(jobState);
        vb.item(jobState.getAction(), RecalculateCampaignsStatusJobParameters.ACTION_LABEL).check(notNull());
        vb.item(jobState.getRecalculationDepth(), RecalculateCampaignsStatusJobParameters.RECALCULATION_DEPTH_LABEL)
                .check(notNull());
        vb.item(jobState.isAllShards(), RecalculateCampaignsStatusJobParameters.ALL_SHARDS_LABEL).check(notNull());
        if (!jobState.isAllShards()) {
            vb.item(jobState.getShard(), RecalculateCampaignsStatusJobParameters.SHARD_LABEL).check(notNull())
                    .check(inRange(1, shardHelper.dbShards().size()));
        }
        return vb.getResult();
    }

    @Override
    public InternalToolResult process(RecalculateCampaignsStatusJobParameters jobParameters) {
        var action = jobParameters.getAction();
        var recalculationDepth = jobParameters.getRecalculationDepth();
        if (jobParameters.isAllShards()) {
            for (var shard : shardHelper.dbShards()) {
                processForSingleShard(shard, action, recalculationDepth);
            }
        } else {
            var shard = jobParameters.getShard();
            processForSingleShard(shard, action, recalculationDepth);
        }

        return new InternalToolResult().withMessage("OK");
    }

    private void processForSingleShard(int shard, RecalculateCampaignsStatusJobParameters.Action action,
                                       RecalculateCampaignsStatusJobParameters.RecalculationDepth recalculationDepth) {
        var lastProcessedCampaignIdPropertyName = lastProcessedCampaignId(shard).getName();
        var jobEnabledPropertyName = recalculateCampaignsStatusJobEnabled(shard).getName();

        ppcPropertiesSupport.set(RECALCULATE_CAMPAIGNS_STATUS_JOB_DEPTH.getName(),
                recalculationDepth.getValue().value());

        switch (action) {
            case NEW_START:
                ppcPropertiesSupport.remove(lastProcessedCampaignIdPropertyName);
                ppcPropertiesSupport.set(jobEnabledPropertyName, String.valueOf(true));
                break;
            case CONTINUE:
                ppcPropertiesSupport.set(jobEnabledPropertyName, String.valueOf(true));
                break;
            case PAUSE:
                ppcPropertiesSupport.set(jobEnabledPropertyName, String.valueOf(false));
                break;
        }
    }
}
