package ru.yandex.direct.internaltools.tools.aggregatedstatuses;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesService;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RecalculationDepthEnum;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.aggregatedstatuses.model.RecomputeCampaignAggregatedStatusesParameters;
import ru.yandex.direct.internaltools.tools.aggregatedstatuses.model.RecomputeCampaignAggregatedStatusesResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.internaltools.tools.aggregatedstatuses.model.RecomputeCampaignAggregatedStatusesResult.error;
import static ru.yandex.direct.internaltools.tools.aggregatedstatuses.model.RecomputeCampaignAggregatedStatusesResult.ok;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;

@Tool(
        name = "Перерасчет агрегированных статусов",
        label = "recompute_aggr_statuses",
        description = "НЕ ИСПОЛЬЗОВАТЬ (КРОМЕ БЕТ), НЕ РАЗОБРАВШИСЬ"
                + " ПОЧЕМУ ПОБИЛИСЬ ДАННЫЕ, ЭТО МОЖЕТ БЫТЬ РЕАЛЬНЫЙ БАГ!!"
                + " По списку Id кампаний пересчитывает статусы кампаний, объявлений, групп, фраз и ретаргетингов",
        consumes = RecomputeCampaignAggregatedStatusesParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.NEW_INTERFACE)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class RecomputeAggregatedStatusesTool extends MassInternalTool<RecomputeCampaignAggregatedStatusesParameters,
        RecomputeCampaignAggregatedStatusesResult> {
    @Autowired
    AggregatedStatusesService aggregatedStatusesService;
    @Autowired
    ShardHelper shardHelper;

    @Override
    public ValidationResult<RecomputeCampaignAggregatedStatusesParameters, Defect> validate(
            RecomputeCampaignAggregatedStatusesParameters recomputeCampaignAggregatedStatusesParameters) {
        ItemValidationBuilder<RecomputeCampaignAggregatedStatusesParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(recomputeCampaignAggregatedStatusesParameters);
        validationBuilder
                .item(recomputeCampaignAggregatedStatusesParameters.getCampaignIds(), "campaignIds")
                .check(isStringWithValidLongIds(1));
        return validationBuilder.getResult();
    }

    @Override
    protected List<RecomputeCampaignAggregatedStatusesResult> getMassData(
            RecomputeCampaignAggregatedStatusesParameters parameter) {
        var updateBefore = LocalDateTime.now();
        Set<Long> campaignIds = getLongIdsFromString(parameter.getCampaignIds());
        var result = new ArrayList<RecomputeCampaignAggregatedStatusesResult>();

        shardHelper.groupByShard(campaignIds, ShardKey.CID).forEach((shard, shardCampaignIds) -> {
            try {
                aggregatedStatusesService.fullyRecalculateStatuses(shard, updateBefore,
                        new HashSet<>(shardCampaignIds), RecalculationDepthEnum.ALL);
                result.addAll(shardCampaignIds.stream().map(i -> ok(i, shard)).collect(toList()));
            } catch (RuntimeException e) {
                // пересчет чанка по одному id, дабы "нормальные" кампании отработали, а по сломанным собралось инфо
                result.addAll(separateDataRecalculation(shard, shardCampaignIds, updateBefore));
            }
        });

        return result;
    }

    private List<RecomputeCampaignAggregatedStatusesResult> separateDataRecalculation(int shard,
                                                                                      Collection<Long> campaignIds,
                                                                                      LocalDateTime updateBefore) {
        var result = new ArrayList<RecomputeCampaignAggregatedStatusesResult>();
        for (var campaignId : campaignIds) {
            try {
                aggregatedStatusesService.fullyRecalculateStatuses(shard, updateBefore,
                        Set.of(campaignId), RecalculationDepthEnum.ALL);
                result.add(ok(campaignId, shard));
            } catch (RuntimeException e) {
                result.add(error(campaignId, e.getMessage()));
            }
        }
        return result;
    }
}
