package ru.yandex.direct.internaltools.tools.asap;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.client.model.ClientsOptions;
import ru.yandex.direct.core.entity.client.repository.ClientOptionsRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.RequiredFeature;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.asap.model.SetAsSoonAsPossibleToolParams;
import ru.yandex.direct.internaltools.tools.asap.model.SetAsSoonAsPossibleToolResult;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptySet;
import static java.util.Objects.nonNull;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userNotFound;
import static ru.yandex.direct.internaltools.tools.asap.SetAsSoonAsPossibleToolUtils.AS_SOON_AS_POSSIBLE_FLAG_VALUE;
import static ru.yandex.direct.internaltools.tools.asap.SetAsSoonAsPossibleToolUtils.buildNewClientFlags;
import static ru.yandex.direct.internaltools.tools.asap.model.SetAsSoonAsPossibleToolParams.ASAP_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.asap.model.SetAsSoonAsPossibleToolParams.LOGINS_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.asap.model.SetAsSoonAsPossibleToolParams.LOGINS_TEXT_FIELD_NAME;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty2;

@Tool(
        name = "Включить клиентам приоритетную отправку",
        label = "set_as_soon_as_possible",
        description = "Включение и выключение приоритетной отправки кампаний для клиентов.\n" +
                "В поле \"Логины пользователей\" нужно ввести логины пользователей через запятую.",
        consumes = SetAsSoonAsPossibleToolParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.OTHER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.MANAGER, InternalToolAccessRole.SUPPORT,
        InternalToolAccessRole.SUPERREADER})
@RequiredFeature(FeatureName.AS_SOON_AS_POSSIBLE)
@ParametersAreNonnullByDefault
public class SetAsSoonAsPossibleTool extends MassInternalTool<SetAsSoonAsPossibleToolParams,
        SetAsSoonAsPossibleToolResult> {

    private final ShardHelper shardHelper;
    private final ClientOptionsRepository clientOptionsRepository;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public SetAsSoonAsPossibleTool(ShardHelper shardHelper, ClientOptionsRepository clientOptionsRepository,
                                   DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.clientOptionsRepository = clientOptionsRepository;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    public ValidationResult<SetAsSoonAsPossibleToolParams, Defect> validate(SetAsSoonAsPossibleToolParams parameters) {
        ItemValidationBuilder<SetAsSoonAsPossibleToolParams, Defect> vb =
                ItemValidationBuilder.of(parameters, Defect.class);

        vb.item(parameters.getLoginsToSetAsap(), LOGINS_TEXT_FIELD_NAME)
                .check(notNull());

        vb.item(parameters.getSetAsap(), ASAP_FIELD_NAME)
                .check(notNull());

        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        return validateLogins(parameters);
    }

    private ValidationResult<SetAsSoonAsPossibleToolParams, Defect> validateLogins(SetAsSoonAsPossibleToolParams parameters) {
        ItemValidationBuilder<SetAsSoonAsPossibleToolParams, Defect> vb =
                ItemValidationBuilder.of(parameters, Defect.class);

        List<String> allLogins = StreamEx.of(parameters.getLoginsToSetAsap().split(","))
                .map(String::trim)
                .distinct()
                .toImmutableList();

        Set<String> existingClientLogins = EntryStream.of(shardHelper.getClientIdsByLogins(allLogins))
                .filterValues(Objects::nonNull)
                .keys()
                .toImmutableSet();

        vb.list(allLogins, LOGINS_FIELD_NAME)
                .checkEach(notEmpty2())
                .checkEach(inSet(existingClientLogins), userNotFound(), When.isValid());

        return vb.getResult();
    }

    @Override
    protected List<SetAsSoonAsPossibleToolResult> getMassData(SetAsSoonAsPossibleToolParams parameters) {
        boolean setAsap = parameters.getSetAsap();
        Set<String> allLogins = StreamEx.of(parameters.getLoginsToSetAsap().split(","))
                .map(String::trim)
                .toSet();

        return processAsapInAllShards(allLogins, setAsap);
    }

    private List<SetAsSoonAsPossibleToolResult> processAsapInAllShards(Collection<String> allLogins, boolean setAsap) {
        Map<String, ClientId> clientIdByLoginAll =
                EntryStream.of(shardHelper.getClientIdsByLogins(mapList(allLogins, identity())))
                        .mapValues(ClientId::fromLong)
                        .toMap();

        return shardHelper.groupByShard(allLogins, ShardKey.LOGIN)
                .stream()
                .mapKeyValue(
                        (shard, loginsInShard) -> processAsapInShard(shard, loginsInShard, setAsap, clientIdByLoginAll))
                .flatMap(StreamEx::of)
                .toList();
    }

    private List<SetAsSoonAsPossibleToolResult> processAsapInShard(int shard, List<String> logins, boolean setAsap,
                                                                   Map<String, ClientId> clientIdByLoginAll) {
        List<ClientId> clientIds = mapList(logins, clientIdByLoginAll::get);

        updateAsapInShard(shard, clientIds, setAsap);
        return fetchToolResultsInShard(shard, logins, clientIds, clientIdByLoginAll);
    }

    private void updateAsapInShard(int shard, List<ClientId> clientIds, boolean setAsap) {
        dslContextProvider.ppcTransaction(shard, configuration -> {
            DSLContext dslContext = configuration.dsl();

            List<ClientsOptions> clientsOptions = clientOptionsRepository.getClientsOptions(dslContext,
                    clientIds, true);

            List<AppliedChanges<ClientsOptions>> appliedChanges = StreamEx.of(clientsOptions)
                    .map(clientOptions -> new ModelChanges<>(clientOptions.getId(), ClientsOptions.class)
                            .process(buildNewClientFlags(clientOptions.getClientFlags(), setAsap),
                                    ClientsOptions.CLIENT_FLAGS)
                            .applyTo(clientOptions))
                    .toList();

            clientOptionsRepository.updateClientsFlags(dslContext, appliedChanges);
        });
    }

    private List<SetAsSoonAsPossibleToolResult> fetchToolResultsInShard(int shard, List<String> logins,
                                                                        List<ClientId> clientIds,
                                                                        Map<String, ClientId> clientIdByLoginAll) {
        Map<ClientId, Set<String>> updatedClientFlagsByClientId =
                StreamEx.of(clientOptionsRepository.getClientsOptions(shard, clientIds))
                        .filter(clientOptions -> nonNull(clientOptions.getClientFlags()))
                        .mapToEntry(clientOptions -> ClientId.fromLong(clientOptions.getId()),
                                ClientsOptions::getClientFlags)
                        .toMap();

        return StreamEx.of(logins)
                .mapToEntry(identity(), clientIdByLoginAll::get)
                .mapValues(clientId ->
                        nvl(updatedClientFlagsByClientId.get(clientId), emptySet()).contains(AS_SOON_AS_POSSIBLE_FLAG_VALUE))
                .map(loginAndAsap -> new SetAsSoonAsPossibleToolResult(loginAndAsap.getKey(), loginAndAsap.getValue()))
                .toList();
    }
}
