package ru.yandex.direct.internaltools.tools.balanceaggrmigration;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.balanceaggrmigration.container.BalanceAggregateMigrationParams;
import ru.yandex.direct.core.entity.campaign.model.AggregatingSumStatus;
import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.core.entity.ppcproperty.model.WalletMigrationStateFlag;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.entity.wallet.model.WalletParamsModel;
import ru.yandex.direct.core.entity.walletparams.repository.WalletParamsRepository;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.balanceaggrmigration.container.BalanceAggregateMigrationParameters;
import ru.yandex.direct.internaltools.tools.balanceaggrmigration.container.BalanceAggregateMigrationResponse;
import ru.yandex.direct.internaltools.tools.balanceaggrmigration.container.MigrationFlagReportStatus;
import ru.yandex.direct.internaltools.tools.balanceaggrmigration.container.MigrationTypeReportValue;
import ru.yandex.direct.internaltools.utils.ToolParameterUtils;
import ru.yandex.direct.utils.DateTimeUtils;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.BALANCE_AGGREGATE_MIGRATION;
import static ru.yandex.direct.internaltools.tools.feature.InternalToolsConstants.LOGINS_SEPARATED_BY_COMMAS_PATTERN;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;

@Tool(
        name = "Управление миграцией общих счетов на новую схему",
        label = "balance_aggregate_migration",
        description = "Интерфейс для включения/выключения миграции"
                + " и добавления клиентов на миграцию общих счетов на обработку баланса по новой схеме",
        consumes = BalanceAggregateMigrationParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.MIGRATION)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER})
@ParametersAreNonnullByDefault
public class BalanceAggregateMigrationTool extends MassInternalTool<BalanceAggregateMigrationParameters,
        BalanceAggregateMigrationResponse> {

    private static final Logger logger = LoggerFactory.getLogger(BalanceAggregateMigrationTool.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DbQueueRepository dbQueueRepository;
    private final WalletRepository walletRepository;
    private final WalletParamsRepository walletParamsRepository;
    private final UserRepository userRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public BalanceAggregateMigrationTool(PpcPropertiesSupport ppcPropertiesSupport,
                                         DbQueueRepository dbQueueRepository,
                                         WalletRepository walletRepository,
                                         WalletParamsRepository walletParamsRepository,
                                         UserRepository userRepository, ShardHelper shardHelper) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.dbQueueRepository = dbQueueRepository;
        this.walletRepository = walletRepository;
        this.walletParamsRepository = walletParamsRepository;
        this.userRepository = userRepository;
        this.shardHelper = shardHelper;
    }

    @Override
    public ValidationResult<BalanceAggregateMigrationParameters, Defect> validate(
            BalanceAggregateMigrationParameters param) {
        Predicate<BalanceAggregateMigrationParameters> paramsAreConsistent =
                p -> isEmpty(p.getLogins()) == (p.getType() == null);

        ItemValidationBuilder<BalanceAggregateMigrationParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(param);

        validationBuilder
                .check(fromPredicate(paramsAreConsistent, inconsistentState()));

        validationBuilder
                .item(param.getLogins(), "logins")
                .check(matchPattern(LOGINS_SEPARATED_BY_COMMAS_PATTERN), When.notNull());

        return validationBuilder.getResult();
    }

    @Override
    protected List<BalanceAggregateMigrationResponse> getMassData() {
        WalletMigrationStateFlag stateFlag = readPpcPropertyWalletMigrationStateFlag();

        MigrationFlagReportStatus flag = stateFlag.isEnabled()
                ? MigrationFlagReportStatus.ENABLED
                : MigrationFlagReportStatus.DISABLED;

        LocalDateTime dateTime = stateFlag.getTime() != null
                ? DateTimeUtils.fromEpochMillis(TimeUnit.SECONDS.toMillis(stateFlag.getTime()))
                : null;

        return singletonList(new BalanceAggregateMigrationResponse(flag, dateTime));
    }

    @Override
    protected List<BalanceAggregateMigrationResponse> getMassData(BalanceAggregateMigrationParameters parameter) {
        boolean migrationFlagCheckbox = nvl(parameter.getEnableMigration(), false);

        WalletMigrationStateFlag dbStateFlag = readPpcPropertyWalletMigrationStateFlag();
        if (dbStateFlag.isEnabled() != migrationFlagCheckbox) {
            WalletMigrationStateFlag stateFlag = new WalletMigrationStateFlag()
                    .withEnabled(migrationFlagCheckbox)
                    .withTime(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()));

            String jsonProperty = JsonUtils.toJson(stateFlag);
            ppcPropertiesSupport.set(PpcPropertyEnum.WALLET_SUMS_MIGRATION_STATE.getName(), jsonProperty);
        }

        if (!isEmpty(parameter.getLogins())) {
            Set<String> logins = ToolParameterUtils.parseCommaSeparatedString(parameter.getLogins());
            createMigratingJobs(logins, parameter.getType());
        }

        return getMassData();
    }


    private WalletMigrationStateFlag readPpcPropertyWalletMigrationStateFlag() {
        return ppcPropertiesSupport.find(PpcPropertyEnum.WALLET_SUMS_MIGRATION_STATE.getName())
                .map(prop -> JsonUtils.fromJson(prop, new TypeReference<WalletMigrationStateFlag>() {
                }))
                .orElse(WalletMigrationStateFlag.disabled());
    }


    private void createMigratingJobs(Collection<String> logins, MigrationTypeReportValue typeReportValue) {
        List<Long> uids = shardHelper.getUidsByLogin(new ArrayList<>(logins));
        List<Long> filteredUids = filterList(uids, Objects::nonNull);
        ShardedData<Long> shardedData = shardHelper.groupByShard(filteredUids, ShardKey.UID);

        shardedData.forEach((shard, userIds) -> {
            List<User> users = userRepository.fetchByUids(shard, userIds);
            for (User user : users) {
                ClientId clientId = user.getClientId();
                List<Wallet> wallets =
                        walletRepository.getAllWalletExistingCampByClientId(shard, singletonList(clientId));
                List<WalletParamsModel> walletParams =
                        walletParamsRepository.get(shard, mapList(wallets, Wallet::getWalletCampaignId));
                Map<Long, WalletParamsModel> walletParamsById =
                        StreamEx.of(walletParams).mapToEntry(WalletParamsModel::getId, identity())
                                .toMap();

                insertWalletMigratingJobsToDbQueue(shard, clientId, user.getUid(), wallets, walletParamsById,
                        typeReportValue);
            }
        });
    }


    private void insertWalletMigratingJobsToDbQueue(int shard, ClientId clientId, long uid, List<Wallet> wallets,
                                                    Map<Long, WalletParamsModel> walletParamsById,
                                                    MigrationTypeReportValue typeReportValue) {
        for (Wallet wallet : wallets) {
            long walletId = wallet.getWalletCampaignId();

            WalletParamsModel walletParams = walletParamsById.getOrDefault(walletId, null);
            String error = validateWallet(wallet, walletParams, typeReportValue);
            if (error != null) {
                logger.info("ignoring wallet {} because {}", walletId, error);
                continue;
            }

            BalanceAggregateMigrationParams params = new BalanceAggregateMigrationParams()
                    .withWalletId(walletId)
                    .withIsRollback(typeReportValue == MigrationTypeReportValue.ROLLBACK);

            Long jobId = dbQueueRepository.insertJob(shard, BALANCE_AGGREGATE_MIGRATION, clientId, uid, params).getId();

            logger.info("job {} was inserted in db queue. clientId: {}; uid: {}; walletId: {}, type {};",
                    jobId, clientId, uid, walletId, typeReportValue);
        }
    }

    /**
     * Проверяет, что ОС можно мигрировать в ту сторону, которую указал пользователь.
     *
     * @return текст ошибки, если этот ОС нельзя мигрировать. null, если можно.
     */
    @Nullable
    private String validateWallet(Wallet wallet, @Nullable WalletParamsModel walletParams,
                                  MigrationTypeReportValue migrationDirection) {
        if (wallet.getCampaignsCurrency().getCode() == CurrencyCode.YND_FIXED) {
            return "wallet currency is YND_FIXED";
        }
        if (walletParams == null) {
            return "wallet params were not found";
        }
        if (
                walletParams.getAggregateMigrateStatus() == AggregatingSumStatus.YES
                        && migrationDirection == MigrationTypeReportValue.MIGRATION
                        || walletParams.getAggregateMigrateStatus() == AggregatingSumStatus.NO
                        && migrationDirection == MigrationTypeReportValue.ROLLBACK) {
            return "invalid migration direction";
        }
        return null;
    }
}

