package ru.yandex.direct.internaltools.tools.banner;

import java.util.Comparator;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboGallery;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.turbogallery.BannerTurboGalleriesRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.banner.model.BannerTurboGalleryUpdateParams;
import ru.yandex.direct.internaltools.tools.banner.model.IntToolBannerTurboGallery;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static java.util.Comparator.naturalOrder;
import static java.util.Comparator.nullsFirst;
import static ru.yandex.direct.core.entity.banner.type.turbogallery.BannerTurboGalleryConstraints.validTurboGalleryHref;
import static ru.yandex.direct.internaltools.tools.banner.model.BannerTurboGalleryUpdateParams.Action.UPDATE_TURBO_GALLERY_HREF;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Tool(
        name = "Добавление и удаление ссылок на Турбо-галерею",
        label = "modify_turbo_gallery",
        description = "Добавление, изменение и удаление ссылки на Турбо-галерею",
        consumes = BannerTurboGalleryUpdateParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.TURBO_GALLERY)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.MANAGER})
@ParametersAreNonnullByDefault
public class BannerTurboGalleryUpdateTool
        extends MassInternalTool<BannerTurboGalleryUpdateParams, IntToolBannerTurboGallery> {

    private final BannerTypedRepository bannerTypedRepository;
    private final BannerTurboGalleriesRepository bannerTurboGalleriesRepository;
    private final BannerModifyRepository bannerModifyRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final CampaignRepository campaignRepository;
    private final FeatureService featureService;
    private final ShardHelper shardHelper;

    public BannerTurboGalleryUpdateTool(BannerTypedRepository bannerTypedRepository,
                                        BannerTurboGalleriesRepository bannerTurboGalleriesRepository,
                                        BannerModifyRepository bannerModifyRepository,
                                        BannerRelationsRepository bannerRelationsRepository,
                                        CampaignRepository campaignRepository,
                                        FeatureService featureService, ShardHelper shardHelper) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerTurboGalleriesRepository = bannerTurboGalleriesRepository;
        this.bannerModifyRepository = bannerModifyRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
        this.shardHelper = shardHelper;
    }

    @Override
    public ValidationResult<BannerTurboGalleryUpdateParams, Defect> validate(BannerTurboGalleryUpdateParams params) {
        ItemValidationBuilder<BannerTurboGalleryUpdateParams, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getBannerId(), "banner_id")
                .check(notNull())
                .check(validId(), When.isValid());
        switch (params.getAction()) {
            case UPDATE_TURBO_GALLERY_HREF:
                vb.item(params.getTurboGalleryHref(), "turbo_gallery_href")
                        .check(notNull())
                        .check(validTurboGalleryHref(), When.isValid());
                break;
            case DELETE_TURBO_GALLERY_HREF:
                vb.item(params.getTurboGalleryHref(), "turbo_gallery_href")
                        .check(isNull());
                break;
        }
        return vb.getResult();
    }

    @Override
    protected List<IntToolBannerTurboGallery> getMassData(BannerTurboGalleryUpdateParams params) {
        Long bannerId = params.getBannerId();
        String turboGalleryHref = params.getAction() == UPDATE_TURBO_GALLERY_HREF ? params.getTurboGalleryHref() : null;

        int shard = shardHelper.getShardByBannerId(bannerId);
        Map<Long, ClientId> clientIdsByBannerIds = bannerRelationsRepository.getClientIdsByBannerIds(shard,
                singletonList(bannerId));
        if (!clientIdsByBannerIds.containsKey(bannerId)) {
            throw new InternalToolValidationException("Incorrect bannerId");
        }
        ClientId clientId = clientIdsByBannerIds.get(bannerId);
        Map<Long, CampaignWithType> campaignsWithType =
                campaignRepository.getCampaignsWithTypeByBannerIds(shard, clientId, singletonList(bannerId));
        CampaignType campaignType = campaignsWithType.get(bannerId).getType();
        if (!campaignType.equals(CampaignType.TEXT)) {
            throw new InternalToolValidationException("Incorrect campaignType: " + campaignType);
        }

        if (!featureService.isEnabledForClientId(clientId, FeatureName.TURBO_FOR_SERP_WIZARD)) {
            throw new InternalToolValidationException(
                    "Feature 'TURBO_FOR_SERP_WIZARD' not enabled for client " + clientId);
        }

        var banners = bannerTypedRepository.getTyped(shard, singletonList(bannerId));
        if (banners.isEmpty() || !(banners.get(0) instanceof BannerWithTurboGallery)) {
            throw new InternalToolValidationException(
                    "banner " + bannerId + " does not have turbo gallery href parameter");

        }
        var banner = (BannerWithTurboGallery) banners.get(0);
        ModelChanges<BannerWithTurboGallery> changesToNewBanner =
                new ModelChanges<>(bannerId, BannerWithTurboGallery.class)
                        .process(turboGalleryHref, BannerWithTurboGallery.TURBO_GALLERY_HREF);

        AppliedChanges<BannerWithTurboGallery> appliedChangesToNewBanners =
                changesToNewBanner.applyTo(banner);

        if (appliedChangesToNewBanners.hasActuallyChangedProps()) {
            bannerModifyRepository.update(new BannerRepositoryContainer(shard),
                    singletonList(appliedChangesToNewBanners));
        }
        return getMassData();
    }

    @Nullable
    @Override
    protected List<IntToolBannerTurboGallery> getMassData() {
        Map<Long, String> bannerTurboGalleries = bannerTurboGalleriesRepository.getAllTurboGalleries();
        Map<Long, ClientId> clientIdsByBannerIds =
                bannerRelationsRepository.getClientIdsByBannerIds(bannerTurboGalleries.keySet());
        List<IntToolBannerTurboGallery> result =
                IntToolBannerConverter.toIntToolBannerTurboGalleries(bannerTurboGalleries, clientIdsByBannerIds);
        Comparator<IntToolBannerTurboGallery> comparator = Comparator
                .comparing(IntToolBannerTurboGallery::getClientId, nullsFirst(naturalOrder()))
                .thenComparing(IntToolBannerTurboGallery::getBannerId, nullsFirst(naturalOrder()))
                .thenComparing(IntToolBannerTurboGallery::getTurboGalleryHref, nullsFirst(naturalOrder()));
        return StreamEx.of(result)
                .sorted(comparator)
                .toList();
    }

}
