package ru.yandex.direct.internaltools.tools.banner;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.banner.model.Language;
import ru.yandex.direct.core.entity.banner.repository.BannersToFillLanguageQueueRepository;
import ru.yandex.direct.core.entity.banner.type.language.BannerLanguageRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.banner.model.BannersLanguageOperationName;
import ru.yandex.direct.internaltools.tools.banner.model.BannersLanguageParameters;
import ru.yandex.direct.internaltools.tools.banner.model.BannersLanguageResult;

@Tool(
        name = "Операции над языком на баннерах",
        label = "process_banners_language",
        description = "Операции над языком на баннерах. Принимает на вход список id баннеров, разделенных через запятую.",
        consumes = BannersLanguageParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.LANGUAGE)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.MANAGER})
@ParametersAreNonnullByDefault
public class BannersLanguageTool extends MassInternalTool<BannersLanguageParameters, BannersLanguageResult> {

    private static final int CHUNK_SIZE = 5000;

    private final ShardHelper shardHelper;
    private final BannerLanguageRepository bannerLanguageRepository;
    private final BannersToFillLanguageQueueRepository bannersToFillLanguageQueueRepository;

    @Autowired
    public BannersLanguageTool(ShardHelper shardHelper,
                               BannerLanguageRepository bannerLanguageRepository,
                               BannersToFillLanguageQueueRepository bannersToFillLanguageQueueRepository) {
        this.shardHelper = shardHelper;
        this.bannerLanguageRepository = bannerLanguageRepository;
        this.bannersToFillLanguageQueueRepository = bannersToFillLanguageQueueRepository;
    }

    @Override
    protected List<BannersLanguageResult> getMassData(BannersLanguageParameters parameter) {
        BannersLanguageOperationName operationName = parameter.getOperationName();

        Set<Long> bids = StreamEx.of(parameter.getBids().split(","))
                .map(String::trim)
                .map(Long::parseLong)
                .toSet();

        switch (operationName) {
            case GET_LANGUAGE:
                return getLanguages(bids);
            case RECALC_LANGUAGE:
                return recalcLanguages(bids);
            default:
                throw new IllegalArgumentException("Unsupported operation");
        }
    }

    private List<BannersLanguageResult> getLanguages(Collection<Long> bids) {
        return processBids(bids, false);
    }

    private List<BannersLanguageResult> recalcLanguages(Collection<Long> bids) {
        return processBids(bids, true);
    }

    private List<BannersLanguageResult> processBids(Collection<Long> bids, boolean shouldRecalcLanguage) {
        Map<Long, Language> languageByBid = new HashMap<>(bids.size());

        shardHelper.groupByShard(bids, ShardKey.BID).chunkedBy(CHUNK_SIZE).forEach((shard, bidsChunk) -> {
            if (shouldRecalcLanguage) {
                bannerLanguageRepository.clearBannersLanguage(shard, bidsChunk);
                bannersToFillLanguageQueueRepository.addItems(shard, bidsChunk);
            }
            languageByBid.putAll(bannerLanguageRepository.getLanguageByBid(shard, bidsChunk));
        });

        return EntryStream.of(languageByBid)
                .map(bannerIdToLanguage ->
                        new BannersLanguageResult(bannerIdToLanguage.getKey(), bannerIdToLanguage.getValue()))
                .toList();
    }
}
