package ru.yandex.direct.internaltools.tools.banner;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.href.BannerDomainRepository;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.banner.model.UpdateDomainsParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.constraint.AdmissibleCharsConstraint;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.lang3.StringUtils.reverse;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefUtils.toUnicodeDomain;
import static ru.yandex.direct.validation.defect.StringDefects.admissibleChars;

@Tool(
        name = "Пересчитать и переотправить в БК домены для переданных баннеров",
        label = "update_domains",
        description = "Пересчитывает домены по href для указанных баннеров. " +
                "Затем отправляет их в бк через ленивую очередь",
        consumes = UpdateDomainsParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.BS_EXPORT)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class UpdateDomainsTool implements BaseInternalTool<UpdateDomainsParameters> {
    private static final AdmissibleCharsConstraint ALLOWED_SYMBOLS =
            new AdmissibleCharsConstraint("0123456789,", admissibleChars());

    private final ShardHelper shardHelper;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerDomainRepository bannerDomainRepository;
    private final DomainRepository domainRepository;
    private final BsResyncService bsResyncService;
    private final BannersUrlHelper bannersUrlHelper;

    public UpdateDomainsTool(ShardHelper shardHelper,
                             BannerTypedRepository bannerTypedRepository,
                             BannerDomainRepository bannerDomainRepository,
                             DomainRepository domainRepository,
                             BsResyncService bsResyncService,
                             BannersUrlHelper bannersUrlHelper) {
        this.shardHelper = shardHelper;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerDomainRepository = bannerDomainRepository;
        this.domainRepository = domainRepository;
        this.bsResyncService = bsResyncService;
        this.bannersUrlHelper = bannersUrlHelper;
    }

    @Override
    public ValidationResult<UpdateDomainsParameters, Defect> validate(UpdateDomainsParameters parameters) {
        ItemValidationBuilder<UpdateDomainsParameters, Defect> v = ItemValidationBuilder.of(parameters);
        v.item(parameters.getBannerIds(), "banner_ids")
                .check(StringConstraints.notEmpty2())
                .check(ALLOWED_SYMBOLS);
        return v.getResult();
    }

    @Override
    public InternalToolResult process(UpdateDomainsParameters updateDomainsParameters) {
        Set<Long> bannerIds = StreamEx.split(updateDomainsParameters.getBannerIds(), ",")
                .map(Long::valueOf)
                .toSet();

        shardHelper.groupByShard(bannerIds, ShardKey.BID)
                .chunkedByDefault()
                .forEach((shard, ids) -> {
                    var bannerIdToHref = bannerTypedRepository.getIdToModelSafely(shard, ids, BannerWithHref.class);
                    Map<Long, Domain> bannerIdToHrefWithDomain = EntryStream.of(bannerIdToHref)
                            .mapValues(banner -> {
                                String domain =
                                        toUnicodeDomain(bannersUrlHelper.extractHostFromHrefWithWwwOrNull(banner.getHref()));
                                return new Domain()
                                        .withDomain(domain)
                                        .withReverseDomain(reverse(domain));
                            })
                            .removeKeyValue((bannerId, domain) -> Objects.equals(domain.getDomain(),
                                    bannerIdToHref.get(bannerId).getDomain()))
                            .toMap();

                    var domains = bannerIdToHrefWithDomain.values();
                    var domainToId = domainRepository.addDomains(shard, domains);
                    domains.forEach(domain -> domain.setId(domainToId.get(domain.getDomain())));

                    bannerDomainRepository.changeBannersDomains(shard, bannerIdToHrefWithDomain);
                    Collection<BsResyncItem> bsResyncItems = StreamEx.of(bannerIdToHrefWithDomain.keySet())
                            .map(bannerIdToHref::get)
                            .map((banner) -> new BsResyncItem(BsResyncPriority.MANUAL_DOMAIN_MIRROR_CORRECTION,
                                    banner.getCampaignId(), banner.getId(), null))
                            .toList();
                    bsResyncService.addObjectsToResync(bsResyncItems);
                });
        return new InternalToolResult().withMessage("Domains updated and banners added to bs_resync_queue");
    }
}
