package ru.yandex.direct.internaltools.tools.bidmodifiers;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.bidmodifiers.service.ComplexBidModifierService;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolProcessingException;
import ru.yandex.direct.internaltools.tools.bidmodifiers.model.BidModifiersSetParameters;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.bidmodifiers.ComplexBidModifierConverter;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.ComplexBidModifierWeb;

import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Tool(
        name = "Установить корректировки на кампанию или на группу",
        label = "set_campaign_bid_modifier",
        description = "Установить корректировки на кампанию или на группу",
        consumes = BidModifiersSetParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.TESTING)
@AccessGroup({InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class SetBidModifiersTool implements BaseInternalTool<BidModifiersSetParameters> {

    private final BidModifierService bidModifierService;
    private final ComplexBidModifierService complexBidModifierService;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;

    @Autowired
    public SetBidModifiersTool(BidModifierService bidModifierService,
                               ComplexBidModifierService complexBidModifierService,
                               ShardHelper shardHelper,
                               CampaignRepository campaignRepository,
                               AdGroupRepository adGroupRepository) {
        this.bidModifierService = bidModifierService;
        this.complexBidModifierService = complexBidModifierService;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
    }

    @Override
    public InternalToolResult process(BidModifiersSetParameters bidModifiersSetParameters) {

        ComplexBidModifierWeb complexBidModifierWeb =
                JsonUtils.fromJson(bidModifiersSetParameters.getComplexBidModifierJson(), ComplexBidModifierWeb.class);

        return updateBidModifiers(
                complexBidModifierWeb,
                bidModifiersSetParameters.getCampaignId(),
                bidModifiersSetParameters.getAdGroupId()
        );
    }

    private InternalToolResult updateBidModifiers(
            ComplexBidModifierWeb complexBidModifierWeb,
            long campaignId,
            @Nullable Long adGroupId
    ) {

        int shard = shardHelper.getShardByCampaignId(campaignId);
        CampaignSimple campaign =
                campaignRepository.getCampaignsSimple(shard, singletonList(campaignId)).get(campaignId);
        long clientId = campaign.getClientId();
        long operatorUid = shardHelper.getUidsByClientId(clientId).get(0);

        ComplexBidModifier complexBidModifier = ComplexBidModifierConverter.fromWeb(complexBidModifierWeb);

        Pair<List<BidModifier>, Multimap<Integer, Integer>> flatModifiersInfo =
                complexBidModifierService.convertFromComplexModelsForCampaigns(singletonList(complexBidModifier));

        List<BidModifier> bidModifiers = flatModifiersInfo.getLeft();

        AdGroup adGroupWithType;
        if (adGroupId != null) {
            List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, singleton(adGroupId));
            adGroupWithType = adGroups.get(0);
        } else {
            adGroupWithType = null;
        }

        ValidationResult<List<BidModifier>, Defect> flatValidationResult =
                complexBidModifierService.validateBidModifiersFlat(
                        bidModifiers, flatModifiersInfo.getRight(), flatIndex -> campaign.getType(),
                        flatIndex -> adGroupWithType,
                        ClientId.fromLong(clientId));

        // Если всё же есть ошибки в валидации -- не сохраняем, генерируем ошибку
        if (flatValidationResult.hasAnyErrors()) {
            throw new InternalToolProcessingException("Ошибка: корректировки не прошли валидацию.\n"
                    + toJson(flatValidationResult));
        }

        // Проставляем campaignId и hierarchicalMultiplierId везде
        bidModifiers.forEach(it -> {
            it.setCampaignId(campaignId);
            if (adGroupId != null) {
                it.setAdGroupId(adGroupId);
            }
        });

        // Сохраняем изменения
        bidModifierService.replaceModifiers(ClientId.fromLong(clientId), operatorUid, bidModifiers,
                singleton(new CampaignIdAndAdGroupIdPair()
                        .withCampaignId(campaignId)
                        .withAdGroupId(adGroupId))
        );

        return new InternalToolResult().withMessage("Корректировки установлены");
    }
}
