package ru.yandex.direct.internaltools.tools.brandlift;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.brandSurvey.BrandSurvey;
import ru.yandex.direct.core.entity.brandlift.repository.BrandSurveyRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.brandlift.model.UpdateBrandLiftOperationInfo;
import ru.yandex.direct.internaltools.tools.brandlift.model.UpdateBrandLiftToolParameter;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.dbschema.ppc.tables.BrandSurvey.BRAND_SURVEY;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.parseCommaSeparatedString;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Tool(
        name = "Обновить is_brand_lift_hidden, experiment_id и segment_id для BrandLift",
        label = "update_fields_for_bl",
        description = "Позволяет перезаписать значения is_brand_lift_hidden, experiment_id и segment_id для выбранных BrandLift.",
        consumes = UpdateBrandLiftToolParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.CPM)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class UpdateBrandLift extends MassInternalTool<UpdateBrandLiftToolParameter, UpdateBrandLiftOperationInfo> {
    private final ShardHelper shardHelper;
    private final BrandSurveyRepository brandSurveyRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final DslContextProvider dslContextProvider;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;

    public UpdateBrandLift(ShardHelper shardHelper,
                           BrandSurveyRepository brandSurveyRepository,
                           RetargetingConditionRepository retargetingConditionRepository,
                           DslContextProvider dslContextProvider, CampaignRepository campaignRepository,
                           CampaignTypedRepository campaignTypedRepository) {
        this.shardHelper = shardHelper;
        this.brandSurveyRepository = brandSurveyRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.dslContextProvider = dslContextProvider;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;

    }

    @Override
    protected List<UpdateBrandLiftOperationInfo> getMassData(UpdateBrandLiftToolParameter parameter) {
        Set<String> brandLiftIds = parseCommaSeparatedString(parameter.getBrandSurveyIds());
        Map<Integer, List<BrandSurvey>> brandLiftsByShard = getBrandLiftsByShard(brandLiftIds);
        Map<String, Integer> shardByBrandLiftId = getShardByBrandLiftId(brandLiftsByShard);
        Map<String, BrandSurvey> brandLiftById = getBrandLiftByBrandLiftId(brandLiftsByShard);
        Map<String, Boolean> isBrandLiftHiddenById = getIsBrandLiftHiddenById(brandLiftById.values());
        Map<Long, RetargetingCondition> retCondByRetCondId = getRetargetingConditionsForBl(brandLiftById.values());

        return brandLiftIds.stream().map(bsId -> {
            var brandSurvey = brandLiftById.get(bsId);
            var retargetingCondition = retCondByRetCondId.get(brandSurvey.getRetargetingConditionId());
            var response = new UpdateBrandLiftOperationInfo()
                    .withBrandLiftIds(brandSurvey.getBrandSurveyId())
                    .withStatus(UpdateBrandLiftOperationInfo.Status.ERROR);
            List<Rule> rules =
                    Optional.ofNullable(retargetingCondition)
                            .map(RetargetingConditionBase::getRules)
                            .orElse(Collections.emptyList());
            if (rules.size() != 1) {
                return response;
            } else {
                Rule rule = rules.get(0);
                if (rule == null || rule.getGoals().size() != 1) {
                    return response;
                } else {
                    Long segmentId = rule.getGoals().get(0).getId();
                    Long experimentId = rule.getSectionId();
                    updateBrandSurvey(
                            shardByBrandLiftId.get(brandSurvey.getBrandSurveyId()),
                            brandSurvey.getBrandSurveyId(), experimentId, segmentId,
                            nvl(isBrandLiftHiddenById.get(brandSurvey.getBrandSurveyId()), Boolean.FALSE)
                    );
                    return response.withStatus(UpdateBrandLiftOperationInfo.Status.OK);
                }
            }
        }).collect(Collectors.toList());
    }

    @Override
    protected List<UpdateBrandLiftOperationInfo> getMassData() {
        return shardHelper.dbShards().stream().map(shard ->
                brandSurveyRepository.getBrandSurveysWithEmptyFields(shard).stream()
                        .map(BrandSurvey::getBrandSurveyId)
                        .map(bs -> new UpdateBrandLiftOperationInfo()
                                .withBrandLiftIds(bs)
                                .withStatus(UpdateBrandLiftOperationInfo.Status.NOT_SET))
                        .collect(Collectors.toList())
        ).flatMap(Collection::stream).collect(Collectors.toList());
    }

    @Override
    public ValidationResult<UpdateBrandLiftToolParameter, Defect> validate(
            UpdateBrandLiftToolParameter parameter) {
        var vb = ItemValidationBuilder.of(parameter, Defect.class);
        vb.item(parameter.getBrandSurveyIds(), "brandSurveyIds")
                .check(notNull());
        return vb.getResult();
    }

    /**
     * Сохраняем необходимые изменения в бд
     */
    private void updateBrandSurvey(int shard, String brandSurveyId, Long newExperimentId, Long newSegmentId,
                                   Boolean isBrandLiftHidden) {
        dslContextProvider.ppc(shard)
                .update(BRAND_SURVEY)
                .set(BRAND_SURVEY.EXPERIMENT_ID, newExperimentId)
                .set(BRAND_SURVEY.SEGMENT_ID, newSegmentId)
                .set(BRAND_SURVEY.IS_BRAND_LIFT_HIDDEN, booleanToLong(isBrandLiftHidden))
                .where(BRAND_SURVEY.BRAND_SURVEY_ID.eq(brandSurveyId))
                .execute();
    }

    /**
     * Получаем мапу brandLiftId -> retargetingCondition
     */
    private Map<Long, RetargetingCondition> getRetargetingConditionsForBl(Collection<BrandSurvey> brandLiftByBrandLiftId) {
        Map<Long, BrandSurvey> brandLiftByRetCondId = listToMap(brandLiftByBrandLiftId,
                BrandSurvey::getRetargetingConditionId);
        return shardHelper.dbShards().stream()
                .map(shard -> retargetingConditionRepository.getConditions(shard,
                        new ArrayList<>(brandLiftByRetCondId.keySet())))
                .flatMap(Collection::stream)
                .collect(Collectors.toMap(RetargetingConditionBase::getId, rc -> rc));
    }

    /**
     * Получаем мапу shard -> список BrandSurvey
     */
    private Map<Integer, List<BrandSurvey>> getBrandLiftsByShard(Collection<String> brandLiftIds) {
        return shardHelper.dbShards().stream()
                .collect(Collectors.toMap(shard -> shard,
                        shard -> brandSurveyRepository.getBrandSurveys(shard, brandLiftIds)));
    }

    /**
     * Получаем мапу brandLiftId -> shard, на котором находится brandLift
     */
    private Map<String, Integer> getShardByBrandLiftId(Map<Integer, List<BrandSurvey>> brandLiftsByShard) {
        return StreamEx.of(brandLiftsByShard.entrySet())
                .flatMapToEntry(entry -> entry.getValue().stream()
                        .map(BrandSurvey::getBrandSurveyId)
                        .collect(Collectors.toMap(id -> id, id -> entry.getKey()))
                ).distinct().toMap();
    }

    /**
     * Получаем мапу brandLiftId -> isBrandLiftHidden
     */
    private Map<String, Boolean> getIsBrandLiftHiddenById(Collection<BrandSurvey> brandLifts) {
        // Считаем шард по ClientID - если brandLift скрытый, то его шард может отличаться от шарда, на котором хранится кампания
        Map<String, Integer> clientShardByBrandLiftId = listToMap(brandLifts, BrandSurvey::getBrandSurveyId,
                brandSurvey -> shardHelper.getShardByClientId(ClientId.fromLong(brandSurvey.getClientId())));
        Map<Integer, List<String>> brandLiftsByClientShard =
                EntryStream.of(clientShardByBrandLiftId).invert()
                        .collapseKeys()
                        .filterValues(l -> !l.isEmpty())
                        .toMap();

        return StreamEx.of(brandLiftsByClientShard.entrySet()).flatMapToEntry(entry -> {
            var shard = entry.getKey();
            var bsIds = entry.getValue();
            // Находим кампании для brandLift'ов и берем первую - если на одной brandLift скрытый, то и на остальных тоже должен быть скрытым
            Map<String, Long> campIdByBL = EntryStream.of(campaignRepository.getCampaignsForBrandSurveys(shard, bsIds))
                    .mapValues(l -> l.get(0))
                    .filterValues(Objects::nonNull)
                    .mapValues(Campaign::getId)
                    .toMap();
            // Достаем нужные поля
            Map<String, Boolean> stringCampaignWithBrandLiftMap =
                    listToMap((List<CampaignWithBrandLift>) campaignTypedRepository.getTypedCampaigns(shard,
                                    campIdByBL.values()),
                            CampaignWithBrandLift::getBrandSurveyId,
                            CommonCampaign::getIsBrandLiftHidden);
            return stringCampaignWithBrandLiftMap;
        }).toMap();
    }

    /**
     * Получаем мапу brandLiftId -> BrandLift
     */
    private Map<String, BrandSurvey> getBrandLiftByBrandLiftId(Map<Integer, List<BrandSurvey>> brandLiftsByShard) {
        return brandLiftsByShard.values().stream()
                .flatMap(Collection::stream)
                .distinct()
                .collect(Collectors.toMap(BrandSurvey::getBrandSurveyId, bs -> bs));
    }

}
