package ru.yandex.direct.internaltools.tools.brandlift;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatusRow;
import ru.yandex.direct.core.entity.campaign.model.SurveyStatus;
import ru.yandex.direct.core.entity.campaign.repository.CampaignBrandSurveyYtRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.brandlift.model.UpdateBrandLiftStatusesToolParameter;
import ru.yandex.direct.internaltools.utils.ToolParameterUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeListNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.parseCommaSeparatedString;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Tool(
        name = "Обновить статус для BrandLift",
        label = "update_statuses_for_bl",
        description = "Позволяет перезаписать статусы DRAFT, MODERATION, MODERATION_REJECTED, ACTIVE, COMPLETED для " +
                "выбранного BrandLift.",
        consumes = UpdateBrandLiftStatusesToolParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.CPM)
@AccessGroup({InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class UpdateBrandLiftStatuses implements BaseInternalTool<UpdateBrandLiftStatusesToolParameter> {
    private static final Logger logger = LoggerFactory.getLogger(UpdateBrandLiftStatuses.class);
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final CampaignBrandSurveyYtRepository campaignBrandSurveyYtRepository;
    private static final Set<String> STOP_REASONS = Set.of("LOW_BUDGET", "LOW_TOTAL_BUDGET", "LOW_DAILY_BUDGET",
            "LOW_REACH");
    private static final String STATUS_OF_CANCELED_BY_STOP_REASONS_CAMPAIGN = "UNFEASIBLE";
    private static final String STATUS_OF_CANCELED_BY_MODERATION_CAMPAIGN = "MODERATION_REJECTED";

    public UpdateBrandLiftStatuses(ShardHelper shardHelper,
                                   CampaignRepository campaignRepository,
                                   CampaignBrandSurveyYtRepository campaignBrandSurveyYtRepository) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.campaignBrandSurveyYtRepository = campaignBrandSurveyYtRepository;
    }

    @Override
    public InternalToolResult process(UpdateBrandLiftStatusesToolParameter parameter) {
        try {
            Set<Long> cids = ToolParameterUtils.getLongIdsFromString(parameter.getCids());
            Map<Long, String> mapCidToBrandSurveyID = getBrandSurveysIds(cids);
            if (mapCidToBrandSurveyID.containsValue(null)) {
                return errorNoBrandlift(mapCidToBrandSurveyID);
            }
            Set<String> stopReasons = parseCommaSeparatedString(parameter.getStopReasons());
            stopReasons = stopReasons.stream().map(String::toLowerCase).collect(Collectors.toSet());
            var brandSurveysRows =
                    fromSurveyStatusToRows(mapCidToBrandSurveyID.values(), parameter.getStatus().name(), stopReasons,
                            parameter.getActions());
            campaignBrandSurveyYtRepository.updateBrandSurveyStates(brandSurveysRows);
            return new InternalToolResult("OK");
        } catch (Exception ex) {
            logger.error("Error updating brandSurvey by campaigns", ex);
            return new InternalToolResult("ERROR");
        }
    }

    private InternalToolResult errorNoBrandlift(Map<Long, String> brandLiftIds) {
        List<Long> ids = brandLiftIds.entrySet()
                .stream()
                .filter(entry -> entry.getValue() == null)
                .map(Map.Entry::getKey)
                .collect(toList());
        return new InternalToolResult(String.format("У данных компаний нет брендлифта: %s. Операция не была " +
                "применена", ids));
    }

    private Map<Long, String> getBrandSurveysIds(Set<Long> campaignIds) {
        var mapCidToShard = shardHelper.groupByShard(campaignIds,
                        ShardKey.CID)
                .getShardedDataMap();
        Map<Long, String> result = new HashMap<>();
        mapCidToShard.forEach((shard, cidsByShard) -> result.putAll(campaignRepository
                .getBrandSurveyIdsForCampaigns(shard, cidsByShard)));
        return result;
    }

    @Override
    public ValidationResult<UpdateBrandLiftStatusesToolParameter, Defect> validate(
            UpdateBrandLiftStatusesToolParameter parameter) {
        var vb = ItemValidationBuilder.of(parameter, Defect.class);
        vb.item(parameter.getCids(), "brandSurveyIds")
                .check(notNull());
        if (parameter.getStopReasons() != null && !parameter.getStopReasons().isEmpty()
                && parameter.getActions() != UpdateBrandLiftStatusesToolParameter.Actions.DO_NOTHING) {
            vb.item(parameter.getStatus(), "Статус")
                    .check(Constraint.fromPredicate(this::isUnfeasible, CommonDefects.invalidValue()));
            Set<String> stopReasons = parseCommaSeparatedString(parameter.getStopReasons());
            vb.item(stopReasons, "Причины остановки")
                    .check(Constraint.fromPredicate(this::isStopReasonsCorrect, CommonDefects.invalidValue()));
        }
        return vb.getResult();
    }

    private boolean isUnfeasible(SurveyStatus status) {
        return status.name().equalsIgnoreCase(STATUS_OF_CANCELED_BY_STOP_REASONS_CAMPAIGN);
    }

    private boolean isStopReasonsCorrect(Set<String> stopReasons) {
        stopReasons = stopReasons.stream().map(String::toUpperCase).collect(Collectors.toSet());
        if (stopReasons.contains("LOW_BUDGET")
                && (stopReasons.contains("LOW_TOTAL_BUDGET") || stopReasons.contains("LOW_DAILY_BUDGET"))) {
            return false;
        }
        return STOP_REASONS.containsAll(stopReasons);
    }

    private List<BrandSurveyStatusRow> fromSurveyStatusToRows(Collection<String> brandSurveyIds,
                                                              String surveyStatus,
                                                              Collection<String> stopReasons,
                                                              UpdateBrandLiftStatusesToolParameter.Actions actions) {
        var idToRowMap = campaignBrandSurveyYtRepository.getBrandSurveyStatuseRows(brandSurveyIds).stream()
                .collect(toMap(BrandSurveyStatusRow::getBrandSurveyId, Function.identity()));
        return brandSurveyIds.stream().map(id -> sureveyStatusToRow(id, surveyStatus, stopReasons,
                actions, idToRowMap.get(id))).collect(toList());
    }

    private BrandSurveyStatusRow sureveyStatusToRow(String brandSurveyIds,
                                                    String surveyStatus,
                                                    Collection<String> stopReasons,
                                                    UpdateBrandLiftStatusesToolParameter.Actions actions,
                                                    @Nullable BrandSurveyStatusRow row) {
        if (row == null) {
            row = new BrandSurveyStatusRow()
                    .withBrandSurveyId(brandSurveyIds)
                    .withBasicUplift(defaultTreeMapNode())
                    .withModerationReasons(defaultTreeListNode())
                    .withStopReasons(defaultTreeListNode());
        }
        setStopReasons(surveyStatus, stopReasons, actions, row);
        setModeretionState(surveyStatus, row);
        setSurveyStatus(surveyStatus, row);

        return row;
    }

    private void setModeretionState(String surveyStatus,
                                    BrandSurveyStatusRow row) {
        if (surveyStatus.equals(STATUS_OF_CANCELED_BY_MODERATION_CAMPAIGN)) {
            row.withModerationState("REJECTED");
        } else {
            row.withModerationState("PASSED");
            row.withModerationReasons(defaultTreeListNode());
        }
    }

    private void setSurveyStatus(String surveyStatus, BrandSurveyStatusRow row) {
        surveyStatus = surveyStatus.equals(STATUS_OF_CANCELED_BY_STOP_REASONS_CAMPAIGN)
                || surveyStatus.equals(STATUS_OF_CANCELED_BY_MODERATION_CAMPAIGN)
                ? "Canceled" : surveyStatus;

        row.withState(surveyStatus);
    }

    private void setStopReasons(String surveyStatus,
                                Collection<String> stopReasons,
                                UpdateBrandLiftStatusesToolParameter.Actions actions,
                                BrandSurveyStatusRow row) {
        if (actions == UpdateBrandLiftStatusesToolParameter.Actions.ADD_STOP_REASONS) {
            List<String> newStopReasons = convertListNode(row.getStopReasons(),
                    YTreeNode::stringValue);
            newStopReasons.addAll(stopReasons);
            row.withStopReasons(collectionToTreeNode(newStopReasons));
        } else if (actions == UpdateBrandLiftStatusesToolParameter.Actions.REPLACE_STOP_REASONS) {
            row.withStopReasons(collectionToTreeNode(stopReasons));
        } else if (!surveyStatus.equals(STATUS_OF_CANCELED_BY_STOP_REASONS_CAMPAIGN)) {
            row.withStopReasons(defaultTreeListNode());
        }
    }

    YTreeListNode defaultTreeListNode() {
        YTreeBuilder stringMapBuilder = YTree.listBuilder();
        return stringMapBuilder.buildList();
    }

    YTreeListNode collectionToTreeNode(Collection<String> list) {
        YTreeBuilder stringListBuilder = YTree.listBuilder();
        list.forEach(stringListBuilder::value);
        return stringListBuilder.buildList();
    }

    YTreeMapNode defaultTreeMapNode() {
        YTreeBuilder stringMapBuilder = YTree.mapBuilder();
        return stringMapBuilder.buildMap();
    }

    private static <T> List<T> convertListNode(@Nullable YTreeNode node, Function<YTreeNode, T> mapper) {
        if (!(node instanceof YTreeListNodeImpl)) {
            return emptyList();
        }

        try {
            return node.asList().stream().map(mapper).collect(toList());
        } catch (RuntimeException ex) {
            logger.error("Error reading node for brand survey", ex);
        }

        return emptyList();
    }
}
