package ru.yandex.direct.internaltools.tools.bs.export.queue;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.bs.export.BsExportParametersService;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.bs.export.queue.container.ShardAndWorkersNumInfo;
import ru.yandex.direct.internaltools.tools.bs.export.queue.model.ManageBsExportWorkersNumParameters;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.bs.export.queue.validation.BsExportQueueDefectIds.MANUAL_CONTROL_OF_WORKERS_NUM_IS_NOT_ENABLED;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@ParametersAreNonnullByDefault
abstract class ManageBsExportWorkersNumTool extends MassInternalTool<ManageBsExportWorkersNumParameters, ShardAndWorkersNumInfo> {
    static final String TOOL_NAME = "Управление количеством воркеров ";
    static final String TOOL_DESCRIPTION_PART_WARNING =
            "Для использования инструмента необходимо включить ручное управление в инструменте '" +
                    BsExportWorkersNumManualBalancingTool.TOOL_NAME + "'\n\n";
    static final String TOOL_DESCRIPTION_PART_MAIN = "Инструмент предназначен для изменения количества воркеров типа ";

    private static final int WORKERS_MIN_NUM = 0;
    private final WorkerType workerType;
    private final BsExportParametersService service;
    private final Constraint<Long, Defect> workersNumRangeChecker;

    final int workersMaxNum;

    /**
     * Создается внутренний инструмент для регулирования количества вокеров указанного типа
     * @param service       сервис для получения/задания различных настроек экспорта в БК
     * @param workerType    тип воркера
     */
    ManageBsExportWorkersNumTool(BsExportParametersService service, WorkerType workerType) {
        this.service = service;
        this.workerType = workerType;

        this.workersMaxNum = BsExportParametersService.getKnownWorkersNum(workerType);
        this.workersNumRangeChecker = inRange((long) WORKERS_MIN_NUM, (long) workersMaxNum);
    }

    boolean allowedToManageWorkersNum() {
        return service.isManualMode();
    }

    @Override
    public ValidationResult<ManageBsExportWorkersNumParameters, Defect> validate(
            ManageBsExportWorkersNumParameters parameters) {

        ItemValidationBuilder<ManageBsExportWorkersNumParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);
        validationBuilder
                .item(parameters.getWorkersNum(), ShardAndWorkersNumInfo.WORKERS_NUM_INFO)
                .check(workersNumRangeChecker);

        ValidationResult<ManageBsExportWorkersNumParameters, Defect> result = validationBuilder.getResult();

        if (!allowedToManageWorkersNum()) {
            result.addError(new Defect<>(MANUAL_CONTROL_OF_WORKERS_NUM_IS_NOT_ENABLED));
        }
        return result;
    }

    @Override
    protected List<ShardAndWorkersNumInfo> getMassData() {
        return service.getWorkersNumForAllShards(workerType)
                .entrySet().stream()
                .map(e -> new ShardAndWorkersNumInfo(e.getKey(), e.getValue()))
                .collect(Collectors.toList());
    }

    @Override
    protected List<ShardAndWorkersNumInfo> getMassData(ManageBsExportWorkersNumParameters parameters) {
        if (parameters.isAllShards()) {
            service.setWorkersNumForAllShards(workerType, parameters.getWorkersNum().intValue());
        } else {
            service.setWorkersNumInShard(workerType, parameters.getShard(), parameters.getWorkersNum().intValue());
        }
        return getMassData();
    }
}
