package ru.yandex.direct.internaltools.tools.bs.export.queue;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.YesNo;
import ru.yandex.direct.core.entity.bs.export.BsExportParametersService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.bs.export.queue.container.ParameterAndValue;
import ru.yandex.direct.internaltools.tools.bs.export.queue.model.FullExportParameters;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.common.db.PpcPropertyNames.FULL_LB_EXPORT_LAST_PROCESSED_CAMPAIGN_ID_PREFIX;
import static ru.yandex.direct.internaltools.tools.bs.export.queue.model.FullExportParameters.ALLOW_ROLLING_WORK_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.bs.export.queue.model.FullExportParameters.MAX_CAMPAIGNS_IN_QUEUE_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.bs.export.queue.model.FullExportParameters.MAX_CHUNK_PER_ITERATION_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.bs.export.queue.model.FullExportParameters.MAX_CHUNK_PER_WORKER_FIELD_NAME;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;

@Tool(
        name = "Управление параметрами полной переотправки через LogBroker в БК",
        label = "manage_bs_full_lb_export_parameters",
        description =
                "Отображает и задает параметры мастер-процесса переотправки Директа в БК (FullExportMasterProcess)\n"
                        + "который отвечает за добавление кампаний в \"очередь\" полной переотправки в БК через LogBroker.\n"
                        + "\n"
                        + "Если chunk_per_worker > 0, в качестве максимума кампаний в очереди будет chunk_per_worker * worker_count,\n"
                        + "но все равно не больше, чем max_campaings_in_queue.\n"
                        + "\n"
                        + "Значения последнего обработанного cid по шардам можно посмотреть во внутреннем отчете по ppc_properties,\n"
                        + "префикс имени свойства: " + FULL_LB_EXPORT_LAST_PROCESSED_CAMPAIGN_ID_PREFIX,

        consumes = FullExportParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.BS_EXPORT)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class ManageFullExportParameters extends MassInternalTool<FullExportParameters, ParameterAndValue> {
    private static final Constraint<Long, Defect> NOT_LESS_THAN_ZERO = notLessThan(0L);

    private BsExportParametersService service;

    @Autowired
    public ManageFullExportParameters(BsExportParametersService bsExportParametersService) {
        this.service = bsExportParametersService;
    }

    @Override
    public ValidationResult<FullExportParameters, Defect> validate(FullExportParameters parameters) {
        ItemValidationBuilder<FullExportParameters, Defect> vb = ItemValidationBuilder.of(parameters);

        vb.item(parameters.getMaxCampaignsInQueue(), MAX_CAMPAIGNS_IN_QUEUE_FIELD_NAME)
                .check(NOT_LESS_THAN_ZERO);

        vb.item(parameters.getMaxChunkPerWorker(), MAX_CHUNK_PER_WORKER_FIELD_NAME)
                .check(NOT_LESS_THAN_ZERO);

        vb.item(parameters.getMaxChunkPerIteration(), MAX_CHUNK_PER_ITERATION_FIELD_NAME)
                .check(NOT_LESS_THAN_ZERO);

        return vb.getResult();
    }

    @Nullable
    @Override
    protected List<ParameterAndValue> getMassData() {
        List<ParameterAndValue> result = new ArrayList<>(4);

        result.add(new ParameterAndValue()
                .withParameter(MAX_CAMPAIGNS_IN_QUEUE_FIELD_NAME)
                .withValue(service.getFullExportMaximumCampaignsInQueue()));
        result.add(new ParameterAndValue()
                .withParameter(MAX_CHUNK_PER_WORKER_FIELD_NAME)
                .withValue(service.getFullExportChunkPerWorker()));
        result.add(new ParameterAndValue()
                .withParameter(MAX_CHUNK_PER_ITERATION_FIELD_NAME)
                .withValue(service.getFullExportMaximumChunkPerIteration()));
        result.add(new ParameterAndValue()
                .withParameter(ALLOW_ROLLING_WORK_FIELD_NAME)
                .withValue(service.canFullExportRollingWork()));

        return result;
    }

    @Override
    protected List<ParameterAndValue> getMassData(FullExportParameters parameters) {
        if (parameters.getMaxCampaignsInQueue() != null) {
            service.setFullExportMaximumCampaignsInQueue(parameters.getMaxCampaignsInQueue().intValue());
        }
        if (parameters.getMaxChunkPerIteration() != null) {
            service.setFullExportMaximumChunkPerIteration(parameters.getMaxChunkPerIteration().intValue());
        }
        if (parameters.getMaxChunkPerWorker() != null) {
            service.setFullExportChunkPerWorker(parameters.getMaxChunkPerWorker().intValue());
        }
        if (parameters.getAllowRollingWork() != null) {
            service.setFullExportRollingWork(parameters.getAllowRollingWork() == YesNo.YES);
        }
        return getMassData();
    }
}
