package ru.yandex.direct.internaltools.tools.bs.export.queue;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.bs.export.queue.model.QueueType;
import ru.yandex.direct.core.entity.bs.export.queue.service.BsExportQueueService;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.bsresyncqueue.model.CampaignIdsAndQueueType;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Arrays.asList;
import static ru.yandex.direct.core.entity.bs.export.queue.model.QueueType.DEV1;
import static ru.yandex.direct.core.entity.bs.export.queue.model.QueueType.DEV2;
import static ru.yandex.direct.core.entity.bs.export.queue.model.QueueType.NOSEND;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.BS_EXPORT;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.BS_EXPORT_LIMITED_QUEUES;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.INTERNAL;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.internaltools.tools.bs.export.queue.validation.BsExportQueueDefectIds.LIMITED_BS_EXPORT_QUEUE_TYPES;
import static ru.yandex.direct.internaltools.tools.bsresyncqueue.model.CampaignIdsAndQueueType.CAMPAIGN_IDS_FIELD_NAME;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;

@Tool(
        name = "Перемещение кампаний в специализированные очереди экспорта в БК",
        label = "move_campaigns_to_bs_export_specials",
        description = "Позволяет назначить кампаниям тип очереди, воркерами которого она будет отправлена в БК.\n"
                + "Заменяет значение par_type в таблице bs_export_specials или удаляет из неё записи.\n",
        consumes = CampaignIdsAndQueueType.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.BS_EXPORT)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class MoveCampaignsToSpecialBsExportQueueTool implements BaseInternalTool<CampaignIdsAndQueueType> {

    private static final Set<CampaignType> SUPPORTED_TYPES = new ImmutableSet.Builder<CampaignType>()
            .addAll(BS_EXPORT)
            .addAll(BS_EXPORT_LIMITED_QUEUES)
            .build();

    private static final Set<QueueType> AVAILABLE_QUEUES_FOR_BS_EXPORT_LIMITED = new HashSet<>(asList(null, NOSEND));

    private static final Set<QueueType> AVAILABLE_QUEUES_FOR_INTERNAL = new HashSet<>(asList(null, NOSEND, DEV1, DEV2));

    private final CampaignService campaignService;
    private final BsExportQueueService bsExportQueueService;

    @Autowired
    public MoveCampaignsToSpecialBsExportQueueTool(
            CampaignService campaignService,
            BsExportQueueService bsExportQueueService) {
        this.campaignService = campaignService;
        this.bsExportQueueService = bsExportQueueService;
    }

    @Override
    public InternalToolResult process(CampaignIdsAndQueueType parameters) {
        Set<Long> campaignIds = getLongIdsFromString(parameters.getCampaignIds());
        bsExportQueueService.setQueueType(parameters.getQueueType(), campaignIds);

        if (parameters.isResetTime()) {
            bsExportQueueService.resetTimeFields(campaignIds);
        }
        return new InternalToolResult().withMessage("OK");
    }

    @Override
    public ValidationResult<CampaignIdsAndQueueType, Defect> validate(CampaignIdsAndQueueType parameters) {
        ValidationResult<CampaignIdsAndQueueType, Defect> validationResult = new ValidationResult<>(parameters);
        Defect campaignIdsDefect = isStringWithValidLongIds(1).apply(parameters.getCampaignIds());
        if (campaignIdsDefect != null) {
            validationResult.addError(campaignIdsDefect);
            return validationResult;
        }

        Set<Long> campaignIds = getLongIdsFromString(parameters.getCampaignIds());
        QueueType queueType = parameters.getQueueType();
        Map<Long, CampaignType> campaignsTypes = campaignService.getCampaignsTypes(campaignIds);

        int idx = 0;
        for (Long campaignId : campaignIds) {
            ValidationResult<Long, Defect> campaignValidationResult = validationResult
                    .getOrCreateSubValidationResult(field(CAMPAIGN_IDS_FIELD_NAME), campaignIds)
                    .getOrCreateSubValidationResult(index(idx++), campaignId);

            CampaignType campaignType = campaignsTypes.get(campaignId);
            if (campaignType == null) {
                campaignValidationResult.addError(campaignNotFound());
                continue;
            }

            if (SUPPORTED_TYPES.contains(campaignType)) {
                if (BS_EXPORT_LIMITED_QUEUES.contains(campaignType)
                        && !AVAILABLE_QUEUES_FOR_BS_EXPORT_LIMITED.contains(queueType)) {
                    campaignValidationResult.addError(new Defect<>(LIMITED_BS_EXPORT_QUEUE_TYPES));
                }

                if (INTERNAL.contains(campaignType)
                        && !AVAILABLE_QUEUES_FOR_INTERNAL.contains(queueType)) {
                    campaignValidationResult.addError(new Defect<>(LIMITED_BS_EXPORT_QUEUE_TYPES));
                }
            } else {
                campaignValidationResult.addError(campaignTypeNotSupported());
            }
        }

        return validationResult;
    }
}
