package ru.yandex.direct.internaltools.tools.bsresyncqueue;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.bsresyncqueue.container.AddedCampaignInfo;
import ru.yandex.direct.internaltools.tools.bsresyncqueue.model.AddCampaignsToBsResyncQueueParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;

@Tool(
        name = "Ленивая перепосылка в БК",
        label = "add_cids_to_resync_queue",
        description = "Добавляет в очередь ленивой переотправки в БК кампании/группы/баннеры.\n"
                + "Кампании могут переотправиться позже, чем их группы/баннеры.\n"
                + "Для МКБ добавляет только кампании.\n"
                + "\n"
                + "Принимает директовские номера кампаний (cid, в БК это ExportID) разделенные пробельными символами или запятыми.",
        consumes = AddCampaignsToBsResyncQueueParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.BS_EXPORT)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class AddCampaignsToBsResyncQueueTool extends MassInternalTool<AddCampaignsToBsResyncQueueParameters, AddedCampaignInfo> {

    private final BsResyncService bsResyncService;

    @Autowired
    public AddCampaignsToBsResyncQueueTool(BsResyncService bsResyncService) {
        this.bsResyncService = bsResyncService;
    }

    @Override
    public ValidationResult<AddCampaignsToBsResyncQueueParameters, Defect> validate(
            AddCampaignsToBsResyncQueueParameters parameters) {
        ItemValidationBuilder<AddCampaignsToBsResyncQueueParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);

        validationBuilder
                .item(parameters.getCampaignIds(), "campaignIds")
                .check(isStringWithValidLongIds(1));
        return validationBuilder.getResult();
    }

    @Override
    public List<AddedCampaignInfo> getMassData(AddCampaignsToBsResyncQueueParameters parameters) {
        Set<Long> campaignIds = getLongIdsFromString(parameters.getCampaignIds());

        Map<Long, Long> result =
                bsResyncService.addWholeCampaignsToResync(campaignIds, parameters.getPriority().getBsResyncPriority());

        return result.entrySet().stream()
                .map(entry -> new AddedCampaignInfo(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());
    }
}
