package ru.yandex.direct.internaltools.tools.bsresyncqueue;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.univocity.parsers.common.DataProcessingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.tools.bsresyncqueue.container.AddObjectInfo;
import ru.yandex.direct.internaltools.tools.bsresyncqueue.model.AddObjectsToBsResyncQueueParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;
import static ru.yandex.direct.internaltools.tools.bsresyncqueue.AddObjectsToBsResyncQueueTool.MAX_VALIDATION_ERRORS;
import static ru.yandex.direct.utils.TsvParserUtils.getBeansList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;

@Tool(
        name = "Импорт файла для массовой переотправки в БК (через bs_resync_queue)",
        label = "add_objects_to_resync_queue",
        description =
                "Импорт файла в очередь переотправки в БК. Принимает файл с 4-мя числами на строку, разделёнными символом табуляции."
                        + " Обязателен заголовок в начале файла из cid bid pid priority.\n"
                        + "- если bid = pid = 0 - отправляем кампанию\n"
                        + "- если bid != 0 - отправляем баннер\n"
                        + "- если pid != 0 - отправляем условие\n"
                        + "priority — приоритет в очереди: 0 — нормальный"
                        + ", <0 — приоритет ниже обычного [отправится позже]"
                        + ", >0 — приоритет выше обычного [отправится раньше]"
                        + ", >100 — приоритетная отправка [отправляются вне зависимости от текущей нагрузки на транспорт]\n\n"
                        + "число отображаемых ошибок валидации ограничено первыми " + MAX_VALIDATION_ERRORS
                        + " строками\n"
                        + "индекс в ошибке валидации - соответствуют порядковому номеру строки в файле\n"
                        + "совсем огромные файлы веб-интерфейс не примет, так что придётся делить на части. верхний предел проходит где-то в районе 2 млн строк."
                        + "Для больших переотправок есть ваншот \"ImportToBsResyncQueueFromYtOneshot\", который берет данные из таблицы в YT",
        consumes = AddObjectsToBsResyncQueueParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.BS_EXPORT)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class AddObjectsToBsResyncQueueTool implements BaseInternalTool<AddObjectsToBsResyncQueueParameters> {
    static final int MAX_VALIDATION_ERRORS = 20;

    static final String SUCCESS_MESSAGE_FORMAT =
            "Файл успешно импортирован. Кол-во добавленных записей в очередь совпадает с кол-вом записей в файле: %d";
    static final String FAILED_MESSAGE_FORMAT =
            "Файл импортирован. Кол-во добавленных записей в очередь: %d, не совпадает с кол-вом записей в файле: %d";
    private static final Logger logger = LoggerFactory.getLogger(AddObjectsToBsResyncQueueTool.class);

    private final BsResyncService bsResyncService;

    @Autowired
    public AddObjectsToBsResyncQueueTool(BsResyncService bsResyncService) {
        this.bsResyncService = bsResyncService;
    }

    @Override
    public ValidationResult<AddObjectsToBsResyncQueueParameters, Defect> validate(
            AddObjectsToBsResyncQueueParameters addObjectsToBsResyncQueueParameters) {
        ItemValidationBuilder<AddObjectsToBsResyncQueueParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(addObjectsToBsResyncQueueParameters);

        List<AddObjectInfo> addObjectInfos;
        try {
            addObjectInfos = getBeansList(addObjectsToBsResyncQueueParameters.getFile(), AddObjectInfo.class);
        } catch (DataProcessingException e) {
            throw new InternalToolValidationException(e.getMessage());
        }

        validationBuilder
                .list(addObjectInfos, "file")
                .check(notNull())
                .check(minListSize(1), When.isValid());

        ValidationResult<AddObjectsToBsResyncQueueParameters, Defect> validationResult = validationBuilder.getResult();
        ValidationResult<List<AddObjectInfo>, Defect> subResult =
                validationResult.getOrCreateSubValidationResult(field("file"), addObjectInfos);

        int errors = 0;
        for (int i = 0; i < addObjectInfos.size(); i++) {
            AddObjectInfo info = addObjectInfos.get(i);

            ValidationResult<AddObjectInfo, Defect> infoValidationResult = validateAddObjectInfo(info);
            if (infoValidationResult.hasAnyErrors()) {
                subResult.getOrCreateSubValidationResult(index(i + 2), info).merge(infoValidationResult);
                errors++;
            }

            if (errors >= MAX_VALIDATION_ERRORS) {
                break;
            }
        }
        return validationResult;
    }

    // да, большая нагрузка на GC, зато валидатор удобный
    private static ValidationResult<AddObjectInfo, Defect> validateAddObjectInfo(AddObjectInfo info) {
        ItemValidationBuilder<AddObjectInfo, Defect> validationBuilder = ItemValidationBuilder.of(info);

        validationBuilder.item(info.getCampaignId(), AddObjectInfo.CAMPAIGN_ID_FIELD_NAME)
                .check(notNull())
                .check(validId());
        validationBuilder.item(info.getAdGroupId(), AddObjectInfo.ADGROUP_ID_FIELD_NAME)
                .check(notNull())
                .check(notLessThan(ID_NOT_SET));
        validationBuilder.item(info.getBannerId(), AddObjectInfo.BANNER_ID_FIELD_NAME)
                .check(notNull())
                .check(notLessThan(ID_NOT_SET));
        validationBuilder.item(info.getPriority(), AddObjectInfo.PRIORITY_FIELD_NAME)
                .check(notNull())
                .check(inRange(BsResyncItem.MIN_PRIORITY, BsResyncItem.MAX_PRIORITY));

        return validationBuilder.getResult();
    }

    @Override
    public InternalToolResult process(AddObjectsToBsResyncQueueParameters addObjectsToBsResyncQueueParameters) {
        List<AddObjectInfo> addObjectInfos =
                getBeansList(addObjectsToBsResyncQueueParameters.getFile(), AddObjectInfo.class);

        List<BsResyncItem> bsResyncItems = addObjectInfos.stream()
                .map(o -> new BsResyncItem(o.getPriority(), o.getCampaignId(), o.getBannerId(), o.getAdGroupId()))
                .collect(toList());

        logger.info("Start importing data from file into bs_resync_queue with {} rows", bsResyncItems.size());
        for (BsResyncItem bsResyncItem : bsResyncItems) {
            logger.info("campaignId={} adgroupId={} bannerId={} priority={}", bsResyncItem.getCampaignId(),
                    bsResyncItem.getAdgroupId(), bsResyncItem.getBannerId(), bsResyncItem.getPriority());
        }
        long addedObjectsCount = bsResyncService.addObjectsToResync(bsResyncItems);
        logger.info("Finish importing data from file. Success import rows is {}", addedObjectsCount);

        return new InternalToolResult()
                .withMessage(getMessage(addObjectInfos.size(), addedObjectsCount));
    }

    private static String getMessage(long objectCount, long addedObjectsCount) {
        return objectCount == addedObjectsCount ?
                format(SUCCESS_MESSAGE_FORMAT, addedObjectsCount)
                : format(FAILED_MESSAGE_FORMAT, addedObjectsCount, objectCount);
    }
}
