package ru.yandex.direct.internaltools.tools.bsresyncqueue;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncQueueStat;
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Disclaimers;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.bsresyncqueue.container.ResyncQueueStat;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Lazy
@Tool(
        name = "Размер очереди ленивой переотправки в БК",
        label = "view_bs_resync_queue_stat",
        description = "Отображает распределение очереди ленивой переотправки в БК по приоритетам.",
        consumes = InternalToolParameter.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.BS_EXPORT)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER})
@Disclaimers({"Подсчёт статистики может занимать несколько минут, дождитесь его завершения"})
@ParametersAreNonnullByDefault
public class ViewBsResyncQueueStatTools extends MassInternalTool<InternalToolParameter, ResyncQueueStat> {
    private static final Comparator<ResyncQueueStat> COMPARATOR =
            Comparator.comparing(ResyncQueueStat::getPriority, Comparator.nullsLast(Long::compareTo)).reversed();

    private final BsResyncService service;

    @Autowired
    public ViewBsResyncQueueStatTools(BsResyncService service) {
        this.service = service;
    }

    @Override
    protected List<ResyncQueueStat> getMassData(InternalToolParameter ignored) {
        Collection<BsResyncQueueStat> stat = service.getQueueStatForAllShards();
        List<ResyncQueueStat> result = mapList(stat, ResyncQueueStat::new);

        Optional<BsResyncQueueStat> total = stat.stream().reduce(BsResyncService::mergeStat);
        if (stat.size() > 1) {
            ResyncQueueStat resyncQueueStat = new ResyncQueueStat(total.get().withPriority(null))
                    .withName("Суммарная статистика по всем приоритетам");
            result.add(resyncQueueStat);
        }

        result.sort(COMPARATOR);
        return result;
    }
}
