package ru.yandex.direct.internaltools.tools.campaign;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusnopay;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.campaign.container.PaymentsForLoginInfo;
import ru.yandex.direct.internaltools.tools.campaign.model.PaymentsForLoginParameters;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.feature.InternalToolsConstants.LOGINS_SEPARATED_BY_COMMAS_PATTERN;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.parseLogins;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;

@Tool(
        name = "Запрет/разрешение оплаты на всех кампаниях на логине",
        label = "payments_for_login",
        description = "Массовый запрет (или разрешение) оплаты - на всех кампаниях на логине. "
                + "При скинутой галочке оплата будет запрещена на всех кампаниях клиента, "
                + "при установленной галочке оплата будет разрешена",
        consumes = PaymentsForLoginParameters.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.VIEW_HISTORY)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class PaymentsForLoginTool extends MassInternalTool<PaymentsForLoginParameters, PaymentsForLoginInfo> {

    private final PaymentsForLoginService paymentsForLoginService;
    private final RbacService rbacService;
    private final UserService userService;

    @Autowired
    public PaymentsForLoginTool(PaymentsForLoginService paymentsForLoginService, RbacService rbacService,
                                UserService userService) {
        this.paymentsForLoginService = paymentsForLoginService;
        this.rbacService = rbacService;
        this.userService = userService;
    }

    @Override
    public ValidationResult<PaymentsForLoginParameters, Defect> validate(
            PaymentsForLoginParameters parameters) {
        ItemValidationBuilder<PaymentsForLoginParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);

        validationBuilder
                .item(parameters.getLogins(), "logins")
                .check(notNull())
                .check(matchPattern(LOGINS_SEPARATED_BY_COMMAS_PATTERN), When.notNull());
        return validationBuilder.getResult();
    }

    @Override
    protected List<PaymentsForLoginInfo> getMassData(PaymentsForLoginParameters parameters) {
        Set<String> logins = parseLogins(parameters.getLogins());

        Map<String, Optional<Long>> loginToUid = StreamEx.of(logins)
                .mapToEntry(login -> Optional.ofNullable(userService.getUidByLogin(login)))
                .toMap();

        List<PaymentsForLoginInfo> result = new ArrayList<>(logins.size());
        for (Map.Entry<String, Optional<Long>> loginEntry : loginToUid.entrySet()) {
            Optional<Long> uid = loginEntry.getValue();
            if (uid.isPresent()) {
                Long chiefUid = rbacService.getChief(uid.get());
                CampaignsStatusnopay currentStatusNoPay = CampaignsStatusnopay.No;
                CampaignsStatusnopay newStatusNoPay = CampaignsStatusnopay.Yes;
                if (parameters.getIsPaymentAllowed()) {
                    currentStatusNoPay = CampaignsStatusnopay.Yes;
                    newStatusNoPay = CampaignsStatusnopay.No;
                }
                List<Long> cidsForUpdate =
                        paymentsForLoginService.getCidsByUidAndStatusNoPay(chiefUid, currentStatusNoPay);
                int cidsCount = paymentsForLoginService.updateStatusNoPay(chiefUid, cidsForUpdate, newStatusNoPay);

                result.add(new PaymentsForLoginInfo(loginEntry.getKey(),
                        "Затронуто " + Integer.toString(cidsCount) + " кампаний"));
            } else {
                result.add(new PaymentsForLoginInfo(loginEntry.getKey(), "Пользователь не найден"));
            }
        }
        return result;
    }
}
