package ru.yandex.direct.internaltools.tools.campaign;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.ContentLanguage;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignAdditionalActionsService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.campaign.container.SmartCampaignLanguageOperationName;
import ru.yandex.direct.internaltools.tools.campaign.container.SmartCampaignLanguageParameters;
import ru.yandex.direct.internaltools.tools.campaign.container.SmartCampaignLanguageResult;
import ru.yandex.direct.queryrec.model.Language;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Tool(
        name = "Работа с языками смарт кампаний",
        label = "process_smart_campaigns_language",
        description = "Операции над языком на смарт кампаниях. Принимает на вход список id кампаний, разделенных " +
                "через запятую.\n" +
                "!!Если в списке будут не только смарты, все лишнее отфильтруется!!\n" +
                "Возможные операции:\n" +
                "- GET — получить язык указанных кампаний\n" +
                "- SET — выставить выбранный язык для указанных кампаний\n" +
                "- REMOVE — сбросить язык для указанных кампаний\n",

        consumes = SmartCampaignLanguageParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.LANGUAGE)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.MANAGER})
@ParametersAreNonnullByDefault
public class SmartCampaignLanguageTool extends MassInternalTool<SmartCampaignLanguageParameters,
        SmartCampaignLanguageResult> {

    private static final int CHUNK_SIZE = 1000;

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final CampaignAdditionalActionsService campaignAdditionalActionsService;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public SmartCampaignLanguageTool(ShardHelper shardHelper,
                                     CampaignRepository campaignRepository,
                                     CampaignAdditionalActionsService campaignAdditionalActionsService,
                                     DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.campaignAdditionalActionsService = campaignAdditionalActionsService;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    protected List<SmartCampaignLanguageResult> getMassData(SmartCampaignLanguageParameters parameter) {
        SmartCampaignLanguageOperationName operationName = parameter.getOperationName();

        Set<Long> campaignIds = StreamEx.of(parameter.getCids().split(","))
                .map(String::trim)
                .map(Long::parseLong)
                .toSet();

        switch (operationName) {
            case GET_LANGUAGE:
                return getLanguageByCampaignIds(campaignIds);
            case REMOVE_LANGUAGE:
                return updateLanguageForCampaignIds(campaignIds, null);
            case SET_LANGUAGE:
                return updateLanguageForCampaignIds(campaignIds, parameter.getLanguage());
            default:
                throw new IllegalArgumentException("Unsupported operation");
        }
    }

    private List<SmartCampaignLanguageResult> getLanguageByCampaignIds(Collection<Long> campaignIds) {
        var smartCampaignIds = filterSmartCampaignIds(campaignIds);
        Map<Long, Language> languageByCampaignIds = new HashMap<>(smartCampaignIds.size());
        shardHelper.groupByShard(smartCampaignIds, ShardKey.CID)
                .chunkedBy(CHUNK_SIZE)
                .forEach((shard, campaignIdsChunk) -> languageByCampaignIds.putAll(campaignRepository.getCampaignsLang(shard,
                        campaignIdsChunk)));
        return EntryStream.of(languageByCampaignIds)
                .mapValues(SmartCampaignLanguageTool::fromLanguage)
                .mapKeyValue(SmartCampaignLanguageResult::new)
                .toList();
    }

    private List<SmartCampaignLanguageResult> updateLanguageForCampaignIds(Collection<Long> campaignIds,
                                                                           @Nullable ContentLanguage language) {

        List<Long> smartCampaignIds = filterSmartCampaignIds(campaignIds);
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedBy(CHUNK_SIZE)
                .forEach((shard, campaignIdsChunk) -> {
                    campaignRepository.updateCampaignLang(shard, campaignIdsChunk, language);
                    var campaignIdsWithPriority = listToMap(campaignIdsChunk,
                            Function.identity(),
                            c -> BsResyncPriority.CHANGE_CAMP_CONTENT_LANG);
                    campaignAdditionalActionsService.addBannersToBsResyncQueue(dslContextProvider.ppc(shard),
                            campaignIdsWithPriority);
                });

        return getLanguageByCampaignIds(smartCampaignIds);
    }

    private static String fromLanguage(@Nullable Language language) {
        if (language == null || language == Language.UNKNOWN) {
            return "null";
        }
        String languageString;
        try {
            languageString = ContentLanguage.fromTypedValue(language.getName()).getTypedValue();
        } catch (IllegalArgumentException e) {
            languageString = language.getName() + " (unsupported)";
        }
        return languageString;
    }

    private List<Long> filterSmartCampaignIds(Collection<Long> campaignIds) {
        List<Long> smartCampaignIds = new ArrayList<>();
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedBy(CHUNK_SIZE)
                .forEach((shard, campaignIdsChunk) -> {
                    var campaingsTypesMap = campaignRepository.getCampaignsTypeMap(shard, campaignIdsChunk);
                    smartCampaignIds.addAll(EntryStream.of(campaingsTypesMap)
                            .filterValues(campaignType -> campaignType == CampaignType.PERFORMANCE)
                            .keys()
                            .toList());
                });
        return smartCampaignIds;
    }

}
