package ru.yandex.direct.internaltools.tools.campaign;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.WhenMoneyOnCampaignWas;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.campaign.container.WhenMoneyOnCampaignWasInfo;
import ru.yandex.direct.internaltools.tools.campaign.model.WhenMoneyOnCampaignsWasParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Comparator.comparing;
import static java.util.Comparator.comparingLong;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;

@Tool(
        name = "Когда на кампании были деньги",
        label = "when_money_on_campaigns_was",
        description = "Показывает когда на кампании были деньги для указанного списка кампаний.",
        consumes = WhenMoneyOnCampaignsWasParameters.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.VIEW_HISTORY)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER,
        InternalToolAccessRole.MANAGER, InternalToolAccessRole.SUPPORT, InternalToolAccessRole.PLACER})
@ParametersAreNonnullByDefault
public class WhenMoneyOnCampaignsWasTool extends MassInternalTool<WhenMoneyOnCampaignsWasParameters, WhenMoneyOnCampaignWasInfo> {

    private final CampaignService campaignService;

    @Autowired
    public WhenMoneyOnCampaignsWasTool(CampaignService campaignService) {
        this.campaignService = campaignService;
    }


    @Override
    public ValidationResult<WhenMoneyOnCampaignsWasParameters, Defect> validate(
            WhenMoneyOnCampaignsWasParameters parameters) {
        ItemValidationBuilder<WhenMoneyOnCampaignsWasParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);

        validationBuilder
                .item(parameters.getCampaignIds(), "campaignIds")
                .check(isStringWithValidLongIds(1));
        return validationBuilder.getResult();
    }

    @Override
    protected List<WhenMoneyOnCampaignWasInfo> getMassData(WhenMoneyOnCampaignsWasParameters parameters) {
        Set<Long> campaignIds = getLongIdsFromString(parameters.getCampaignIds());

        List<WhenMoneyOnCampaignWas> whenMoneyOnCampaignsWas = campaignService.getWhenMoneyOnCampaignsWas(campaignIds);

        return whenMoneyOnCampaignsWas.stream()
                .map(r -> new WhenMoneyOnCampaignWasInfo()
                        .withCampaignId(r.getCampaignId())
                        .withIntervalStart(r.getIntervalStart())
                        .withIntervalEnd(r.getIntervalEnd())
                )
                //Сортируем по campaignId. При равных campaignId сортируем по убыванию intervalEnd
                .sorted(comparingLong(WhenMoneyOnCampaignWasInfo::getCampaignId)
                        .thenComparing(comparing(WhenMoneyOnCampaignWasInfo::getIntervalEnd).reversed())
                )
                .collect(Collectors.toList());
    }
}
