package ru.yandex.direct.internaltools.tools.canvas;

import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.canvas.CanvasToolsRequests.AdminRejectRequest;
import ru.yandex.direct.canvas.model.OnCreativeOperationResult;
import ru.yandex.direct.canvas.tools_client.CanvasToolsClient;
import ru.yandex.direct.core.entity.creative.constants.AdminRejectionReason;
import ru.yandex.direct.i18n.Translator;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolParameter;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.core.input.InternalToolInput;
import ru.yandex.direct.internaltools.core.input.InternalToolInputPreProcessor;
import ru.yandex.direct.internaltools.tools.canvas.model.AdminRejectCreativesParameter;
import ru.yandex.direct.internaltools.tools.canvas.model.OnCreativeOperationResultWithId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.i18n.I18NBundle.makeStubTranslatorFactory;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Tool(
        name = "Запрет использования креативов",
        label = "canvas_admin_reject",
        description = "Запрещает использование креативов канваса в баннера. " +
                "Баннеры с уже привязанными креативами останавливает. " +
                "Не забудьте переотправить баннеры в директ после установки флага",
        consumes = AdminRejectCreativesParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.TURN_OFF)
@Category(InternalToolCategory.CANVAS)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class AdminRejectCreativesTool
        extends MassInternalTool<AdminRejectCreativesParameter, OnCreativeOperationResultWithId> {
    private static final String LIFT_REJECTION_STRING = "Снять запрет";
    private static final String REASON_TEXT_TEMPLATE = "%s : %s";
    private static final Pattern REASON_TEXT_PATTERN = Pattern.compile("(.*) : .*");

    @Autowired
    CanvasToolsClient canvasToolsClient;

    @Override
    public ValidationResult<AdminRejectCreativesParameter, Defect> validate(AdminRejectCreativesParameter parameter) {
        ItemValidationBuilder<AdminRejectCreativesParameter, Defect> validationBuilder =
                ItemValidationBuilder.of(parameter);
        validationBuilder
                .item(parameter.getCreativeIds(), "creativeIds")
                .check(isStringWithValidLongIds(1));
        return validationBuilder.getResult();
    }

    @Override
    public List<OnCreativeOperationResultWithId> getMassData(AdminRejectCreativesParameter parameter) {
        Set<Long> creativeIds = getLongIdsFromString(parameter.getCreativeIds());
        Matcher matcher = REASON_TEXT_PATTERN.matcher(parameter.getBlockReason());
        String reason;
        if (matcher.matches()) {
            reason = matcher.group(1);
        } else if (LIFT_REJECTION_STRING.equals(parameter.getBlockReason())) {
            reason = "";
        } else {
            throw new IllegalArgumentException("Неизвестная причина: " + parameter.getBlockReason());
        }
        AdminRejectRequest request = AdminRejectRequest.newBuilder()
                .addAllCreativeIds(creativeIds)
                .setReason(reason)
                .build();
        return mapToResultList(canvasToolsClient.adminReject(request));
    }

    @Component
    public static class BlockReasonPreprocessor implements InternalToolInputPreProcessor<String> {
        @Override
        public <T extends InternalToolParameter> InternalToolInput.Builder<T, String> preSend(
                InternalToolInput.Builder<T, String> inputBuilder) {
            Translator translator = makeStubTranslatorFactory()
                    .getTranslator(new Locale.Builder().setLanguageTag("ru-RU").build());
            List<String> reasons = StreamEx.of(AdminRejectionReason.values())
                    .map(reason -> String.format(REASON_TEXT_TEMPLATE,
                            reason.name(),
                            reason.getReason().translate(translator)))
                    .append(LIFT_REJECTION_STRING)
                    .collect(toList());
            return inputBuilder
                    .withDefaultValue(LIFT_REJECTION_STRING)
                    .withAllowedValues(reasons)
                    .addValidator(inSet(Set.copyOf(reasons)));
        }
    }

    private List<OnCreativeOperationResultWithId> mapToResultList(Map<Long, OnCreativeOperationResult> resultMap) {
        return resultMap.keySet().stream().map(id -> new OnCreativeOperationResultWithId(id, resultMap.get(id)))
                .collect(Collectors.toList());
    }
}
