package ru.yandex.direct.internaltools.tools.canvas;


import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.canvas.tools_client.CanvasToolsClient;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.canvas.model.UpdateVastByIdParameter;
import ru.yandex.direct.internaltools.tools.canvas.model.VastDataResponse;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Tool(
        name = "Изменить VAST из Канваса",
        label = "canvas_edit_vast_operation",
        description = "Меняет VAST в базе Канваса",
        consumes = UpdateVastByIdParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.CANVAS)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class EditVastByIdTool extends MassInternalTool<UpdateVastByIdParameter,
        VastDataResponse> {
    @Autowired
    CanvasToolsClient canvasToolsClient;

    @Override
    public ValidationResult<UpdateVastByIdParameter, Defect> validate(
            UpdateVastByIdParameter updateVastByIdParameter) {

        ItemValidationBuilder<UpdateVastByIdParameter, Defect> validationBuilder =
                ItemValidationBuilder.of(updateVastByIdParameter);

        validationBuilder
                .item(updateVastByIdParameter.getCreativeId(), "creativeId")
                .check(Constraint.fromPredicate(id -> id > 0, CommonDefects.validId()))
                .item(updateVastByIdParameter.getVast(), "vast")
                .check(Constraint.fromPredicate(vast -> vast != null && vast.length > 0, CommonDefects.notNull()));

        return validationBuilder.getResult();
    }

    @Override
    protected List<VastDataResponse> getMassData(UpdateVastByIdParameter parameter) {
        Long creativeId = parameter.getCreativeId();
        String creativeData;
        String vastData = ifNotNull(parameter.getVast(), e -> new String(e, StandardCharsets.UTF_8));

        if (vastData == null || vastData.isBlank()) {
            throw new InternalToolValidationException("Incorrect vast");
        } else {
            creativeData = canvasToolsClient.updateVideoVast(creativeId, vastData);
        }

        return Collections.singletonList(new VastDataResponse(creativeId, creativeData));
    }

}
