package ru.yandex.direct.internaltools.tools.canvas;


import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.canvas.model.OnCreativeOperationResult;
import ru.yandex.direct.canvas.tools_client.CanvasToolsClient;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.canvas.model.OnCreativeIdsOperationParameter;
import ru.yandex.direct.internaltools.tools.canvas.model.OnCreativeOperationResultWithId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;

@Tool(
        name = "Операции над креативами",
        label = "canvas_on_creatives_operation",
        description = "Запускает операцию в Конструкторе Креативов по списку Id креативов",
        consumes = OnCreativeIdsOperationParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.CANVAS)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class OnCreativeIdsOperationsTool extends
        MassInternalTool<OnCreativeIdsOperationParameter, OnCreativeOperationResultWithId> {
    @Autowired
    CanvasToolsClient canvasToolsClient;

    @Override
    public ValidationResult<OnCreativeIdsOperationParameter, Defect> validate(
            OnCreativeIdsOperationParameter onCreativeIdsOperationParameter) {
        ItemValidationBuilder<OnCreativeIdsOperationParameter, Defect> validationBuilder =
                ItemValidationBuilder.of(onCreativeIdsOperationParameter);
        validationBuilder
                .item(onCreativeIdsOperationParameter.getCreativeIds(), "creativeIds")
                .check(isStringWithValidLongIds(1));
        return validationBuilder.getResult();
    }

    @Override
    protected List<OnCreativeOperationResultWithId> getMassData(OnCreativeIdsOperationParameter parameter) {
        Set<Long> creativeIds = getLongIdsFromString(parameter.getCreativeIds());
        switch (parameter.getOperationName()) { // make it 200 cases, i dare you!
            case RESHOOT_SCREENSHOT:
                return mapToResultList(canvasToolsClient.reshootScreenshot(creativeIds));
            case REBUILD:
                return mapToResultList(canvasToolsClient.rebuild(creativeIds));
            case SEND_TO_DIRECT:
                return mapToResultList(canvasToolsClient.sendToDirect(creativeIds));
            case SEND_TO_RTBHOST:
                return mapToResultList(canvasToolsClient.sendToRtbHost(creativeIds));
            default:
                throw new IllegalStateException("Unexpected enum value: " + parameter.getOperationName() + " which " +
                        "has no handler");
        }
    }

    private List<OnCreativeOperationResultWithId> mapToResultList(Map<Long, OnCreativeOperationResult> resultMap) {
        return resultMap.keySet().stream().map(id -> new OnCreativeOperationResultWithId(id, resultMap.get(id)))
                .collect(Collectors.toList());
    }
}
