package ru.yandex.direct.internaltools.tools.cashback.model;

import java.util.Objects;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.cashback.model.CashbackProgram;

import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.toDisplayPercent;

/**
 * Ключ программы кешбэка длля использования в коде и отдачи в интерфейс
 * в виде строки:
 * <pre>
 *     <id программы> - <Название категории \ Название программы (если название категории - null)> <процент>% [публичная] [активная]
 * </pre>
 */
public class CashbackProgramKey {
    private static final Pattern KEY_PATTERN = Pattern.compile("(\\d+)\\s+.*");

    private final String categoryName;
    private final String programName;
    private final Long id;
    private final String percent;
    private final boolean isPublic;
    private final boolean isEnabled;

    public CashbackProgramKey(String categoryName, String programName, Long id, String percent, boolean isPublic, boolean isEnabled) {
        this.categoryName = categoryName;
        this.programName = programName;
        this.id = id;
        this.percent = percent;
        this.isPublic = isPublic;
        this.isEnabled = isEnabled;
    }

    public String toDisplayString() {
        var builder = new StringBuilder()
                .append(id)
                .append(" - ")
                .append(Objects.nonNull(categoryName) ? categoryName : programName)
                .append(" ")
                .append(percent)
                .append("%");
        if (isPublic) {
            builder.append(" ").append("публичная");
        }
        if (isEnabled) {
            builder.append(" ").append("активная");
        }
        return builder.toString();
    }

    public static CashbackProgramKey fromCashbackProgram(CashbackProgram program) {
        return new CashbackProgramKey(
                program.getCategoryNameRu(),
                program.getNameRu(),
                program.getId(),
                toDisplayPercent(program.getPercent()).toPlainString(),
                program.getIsPublic(),
                program.getIsEnabled());
    }

    @Nullable
    public static Long parseId(String keyString) {
        var matcher = KEY_PATTERN.matcher(keyString);
        if (matcher.matches()) {
            return Long.parseLong(matcher.group(1));
        }
        return null;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CashbackProgramKey that = (CashbackProgramKey) o;
        return percent == that.percent &&
                isPublic == that.isPublic &&
                isEnabled == that.isEnabled &&
                categoryName.equals(that.categoryName) &&
                programName.equals(that.programName) &&
                id.equals(that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hash(categoryName, programName, id, percent, isPublic, isEnabled);
    }
}
