package ru.yandex.direct.internaltools.tools.cashback.repository;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.cashback.model.CashbackHistoryAction;
import ru.yandex.direct.core.entity.cashback.model.CashbackHistoryStateChange;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramState;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_CASHBACK_HISTORY;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_CASHBACK_PROGRAMS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class InternalToolsCashbackClientsRepository {
    private final JooqMapperWithSupplier<CashbackProgramState> programStateMapper;
    private final JooqMapperWithSupplier<CashbackHistoryAction> historyMapper;

    @Autowired
    public InternalToolsCashbackClientsRepository() {
        this.programStateMapper = createProgramStateMapper();
        this.historyMapper = createHistoryMapper();
    }

    public Map<ClientId, CashbackProgramState> getClientsCashbackProgramStates(
            DSLContext context, Long programId, List<ClientId> clientIds) {
        var result = context
                .select(programStateMapper.getFieldsToRead())
                .from(CLIENTS_CASHBACK_PROGRAMS)
                .where(CLIENTS_CASHBACK_PROGRAMS.CLIENT_ID.in(clientIds)
                        .and(CLIENTS_CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID.eq(programId)))
                .fetch(programStateMapper::fromDb);
        return listToMap(result, CashbackProgramState::getClientId, Function.identity());
    }

    public void updateClientProgramState(Configuration conf, Long enabled, List<Long> stateIds) {
        if (stateIds.isEmpty()) {
            return;
        }
        conf.dsl().update(CLIENTS_CASHBACK_PROGRAMS)
                .set(CLIENTS_CASHBACK_PROGRAMS.IS_ENABLED, enabled)
                .where(CLIENTS_CASHBACK_PROGRAMS.CLIENT_CASHBACK_PROGRAM_ID.in(stateIds))
                .execute();
    }

    public void createProgramStates(Configuration conf, List<CashbackProgramState> states) {
        new InsertHelper<>(conf.dsl(), CLIENTS_CASHBACK_PROGRAMS)
                .addAll(programStateMapper, states)
                .executeIfRecordsAdded();
    }

    public void addHistoryRecords(Configuration conf, List<CashbackHistoryAction> actions) {
        new InsertHelper<>(conf.dsl(), CLIENTS_CASHBACK_HISTORY)
                .addAll(historyMapper, actions)
                .executeIfRecordsAdded();
    }

    public List<CashbackHistoryAction> getHistoryByClient(DSLContext context, Long programId, ClientId clientId) {
        return context
                .select(historyMapper.getFieldsToRead())
                .from(CLIENTS_CASHBACK_HISTORY)
                .join(CLIENTS_CASHBACK_PROGRAMS)
                .on(CLIENTS_CASHBACK_PROGRAMS.CLIENT_CASHBACK_PROGRAM_ID.eq(CLIENTS_CASHBACK_HISTORY.CLIENT_CASHBACK_PROGRAM_ID))
                .where(CLIENTS_CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID.eq(programId))
                    .and(CLIENTS_CASHBACK_PROGRAMS.CLIENT_ID.eq(clientId.asLong()))
                .orderBy(CLIENTS_CASHBACK_HISTORY.CHANGE_TIME)
                .fetch(historyMapper::fromDb);
    }

    public List<ClientId> collectClientsByProgramId(DSLContext context, Long programId, boolean in) {
        var raw = context
                .select(CLIENTS_CASHBACK_PROGRAMS.CLIENT_ID)
                .from(CLIENTS_CASHBACK_PROGRAMS)
                .where(CLIENTS_CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID.eq(programId)
                        .and(CLIENTS_CASHBACK_PROGRAMS.IS_ENABLED.eq(booleanToLong(in))))
                .fetch(CLIENTS_CASHBACK_PROGRAMS.CLIENT_ID);
        return mapList(raw, ClientId::fromLong);
    }

    private JooqMapperWithSupplier<CashbackProgramState> createProgramStateMapper() {
        return JooqMapperWithSupplierBuilder.builder(CashbackProgramState::new)
                .map(property(CashbackProgramState.CLIENT_PROGRAM_ID, CLIENTS_CASHBACK_PROGRAMS.CLIENT_CASHBACK_PROGRAM_ID))
                .map(property(CashbackProgramState.PROGRAM_ID, CLIENTS_CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID))
                .map(clientIdProperty(CashbackProgramState.CLIENT_ID, CLIENTS_CASHBACK_PROGRAMS.CLIENT_ID))
                .map(booleanProperty(CashbackProgramState.IS_ENABLED, CLIENTS_CASHBACK_PROGRAMS.IS_ENABLED))
                .build();
    }

    private JooqMapperWithSupplier<CashbackHistoryAction> createHistoryMapper() {
        return JooqMapperWithSupplierBuilder.builder(CashbackHistoryAction::new)
                .map(property(CashbackHistoryAction.HISTORY_ID, CLIENTS_CASHBACK_HISTORY.CLIENT_CASHBACK_HISTORY_ID))
                .map(property(CashbackHistoryAction.CLIENT_PROGRAM_ID, CLIENTS_CASHBACK_HISTORY.CLIENT_CASHBACK_PROGRAM_ID))
                .map(convertibleProperty(CashbackHistoryAction.STATE_CHANGE, CLIENTS_CASHBACK_HISTORY.STATE_CHANGE,
                        CashbackHistoryStateChange::fromSource, CashbackHistoryStateChange::toSource))
                .build();
    }
}
