package ru.yandex.direct.internaltools.tools.cashback.service;

import java.util.ArrayList;
import java.util.List;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.cashback.model.CashbackHistoryAction;
import ru.yandex.direct.core.entity.cashback.model.CashbackHistoryStateChange;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgram;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramState;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.tools.cashback.repository.InternalToolsCashbackClientsRepository;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.core.entity.cashback.CashbackConstants.PUBLIC_PROGRAMS_CLIENT_ID;
import static ru.yandex.direct.core.entity.cashback.CashbackConstants.PUBLIC_PROGRAMS_SHARD_ID;
import static ru.yandex.direct.core.entity.cashback.model.CashbackHistoryStateChange.IN;
import static ru.yandex.direct.core.entity.cashback.model.CashbackHistoryStateChange.OUT;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class CashbackClientsWriteService {
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final CashbackNotificationsService notificationsService;
    private final InternalToolsCashbackClientsRepository repository;

    @Autowired
    public CashbackClientsWriteService(ShardHelper shardHelper,
                                       DslContextProvider dslContextProvider,
                                       CashbackNotificationsService notificationsService,
                                       InternalToolsCashbackClientsRepository repository) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.notificationsService = notificationsService;
        this.repository = repository;
    }

    public void addClientsToProgram(User operator, CashbackProgram program, List<ClientId> clientIds) {
        addClientsToProgram(program.getId(), clientIds);
        notificationsService.notifyClientsAddedToProgram(program, clientIds, operator);
    }

    public void removeClientsFromProgram(User operator, CashbackProgram program, List<ClientId> clientIds) {
        removeClientsFromProgram(program.getId(), clientIds);
        notificationsService.notifyClientsRemovedFromProgram(program, clientIds, operator);
    }

    public void enablePublicProgram(CashbackProgram program) {
        enablePublicProgram(program.getId());
    }

    public void disablePublicProgram(CashbackProgram program) {
        disablePublicProgram(program.getId());
    }

    public void disableProgram(User operator, CashbackProgram program) {
        List<ClientId> clientIds = new ArrayList<>();
        shardHelper.forEachShard(shard -> clientIds.addAll(
                repository.collectClientsByProgramId(dslContextProvider.ppc(shard), program.getId(), true)));
        removeClientsFromProgram(operator, program, clientIds);
    }

    public void handleProgramStatusChange(User operator, AppliedChanges<CashbackProgram> change) {
        var model = change.getModel();
        boolean isPublic = nvl(change.getNewValue(CashbackProgram.IS_PUBLIC), false);
        boolean isEnabled = nvl(change.getNewValue(CashbackProgram.IS_ENABLED), false);
        if (change.changed(CashbackProgram.IS_ENABLED)) {
            handleProgramEnabledStatus(operator, model, isEnabled, isPublic);
        }
        if (change.changed(CashbackProgram.IS_PUBLIC)) {
            handleProgramPublicityStatus(operator, model, isPublic);
        }
    }

    private void enablePublicProgram(Long programId) {
        changeProgramStateForClients(PUBLIC_PROGRAMS_SHARD_ID, programId, true, List.of(PUBLIC_PROGRAMS_CLIENT_ID));
    }

    private void disablePublicProgram(Long programId) {
        changeProgramStateForClients(PUBLIC_PROGRAMS_SHARD_ID, programId, false, List.of(PUBLIC_PROGRAMS_CLIENT_ID));
    }

    private void addClientsToProgram(Long programId, List<ClientId> clientIds) {
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, clientIdsOnShard) -> changeProgramStateForClients(shard, programId, true, clientIdsOnShard));
    }

    private void removeClientsFromProgram(Long programId, List<ClientId> clientIds) {
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, clientIdsOnShard) -> changeProgramStateForClients(shard, programId, false, clientIdsOnShard));
    }

    private void changeProgramStateForClients(int shard, Long programId, boolean enabled, List<ClientId> clientIds) {
        var change = enabled ? IN : OUT;
        var clientIdToProgramState =
                repository.getClientsCashbackProgramStates(dslContextProvider.ppc(shard), programId, clientIds);

        var stateIds = mapList(clientIdToProgramState.values(), CashbackProgramState::getClientProgramId);
        dslContextProvider.ppcTransaction(shard, conf -> {
            repository.updateClientProgramState(conf, booleanToLong(enabled), stateIds);
            writeHistory(conf, List.copyOf(clientIdToProgramState.values()), change);

            // А теперь создадим новые записи
            var statesToCreate = StreamEx.of(clientIds)
                    .filter(id -> !clientIdToProgramState.containsKey(id))
                    .map(id -> new CashbackProgramState()
                            .withClientId(id)
                            .withIsEnabled(enabled)
                            .withProgramId(programId))
                    .toList();
            var ids = shardHelper.generateClientsCashbackProgramsIds(statesToCreate.size());
            EntryStream.zip(statesToCreate, ids).forKeyValue(CashbackProgramState::setClientProgramId);
            repository.createProgramStates(conf, statesToCreate);
            writeHistory(conf, statesToCreate, change);
        });
    }

    public CashbackProgramState getPublicProgramState(Long programId) {
        return getProgramState(PUBLIC_PROGRAMS_SHARD_ID, programId, PUBLIC_PROGRAMS_CLIENT_ID);
    }

    public List<CashbackHistoryAction> collectPublicProgramHistory(Long programId) {
        return collectProgramHistory(PUBLIC_PROGRAMS_SHARD_ID, programId, PUBLIC_PROGRAMS_CLIENT_ID);
    }

    public CashbackProgramState getProgramState(ClientId clientId, Long programId) {
        var shard = shardHelper.getShardByClientId(clientId);
        return getProgramState(shard, programId, clientId);
    }

    public List<CashbackHistoryAction> collectProgramHistory(ClientId clientId, Long programId) {
        var shard = shardHelper.getShardByClientId(clientId);
        return collectProgramHistory(shard, programId, clientId);
    }

    private CashbackProgramState getProgramState(int shard, Long programId, ClientId clientId) {
        var states = repository.getClientsCashbackProgramStates(
                dslContextProvider.ppc(shard), programId, List.of(clientId));
        return requireNonNull(states.get(clientId));
    }

    private List<CashbackHistoryAction> collectProgramHistory(int shard, Long programId, ClientId clientId) {
        return repository.getHistoryByClient(dslContextProvider.ppc(shard), programId, clientId);
    }

    private void writeHistory(Configuration conf, List<CashbackProgramState> states, CashbackHistoryStateChange change) {
        var entries = mapList(states, state ->
                new CashbackHistoryAction()
                        .withClientProgramId(state.getClientProgramId())
                        .withStateChange(change));
        var ids = shardHelper.generateClientCashbacksHistoryIds(entries.size());
        EntryStream.zip(entries, ids).forKeyValue(CashbackHistoryAction::setHistoryId);
        repository.addHistoryRecords(conf, entries);
    }

    private void handleProgramEnabledStatus(User operator, CashbackProgram program, boolean isEnabled, boolean isPublic) {
        if (isEnabled) {
            if (isPublic) {
                enablePublicProgram(program);
            }
        } else {
            if (isPublic) {
                disablePublicProgram(program);
            } else {
                disableProgram(operator, program);
            }
        }
    }

    private void handleProgramPublicityStatus(User operator, CashbackProgram program, boolean isPublic) {
        if (isPublic) {
            disableProgram(operator, program);
        } else {
            disablePublicProgram(program);
        }
    }
}
