package ru.yandex.direct.internaltools.tools.cashback.service;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.cashback.model.CashbackCategory;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgram;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramCategory;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramCategoryExtended;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.mail.MailSender;

import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.toDisplayPercent;

@Service
public class CashbackNotificationsService {
    private final MailSender mailSender;

    @Autowired
    public CashbackNotificationsService(MailSender mailSender) {
        this.mailSender = mailSender;
    }

    public void notifyCategoryCreated(CashbackCategory category, User operator) {
        var messageText = "Добавлена новая категория\n" +
                "ID: " + category.getId() + "\n" +
                "Название (ru): " + category.getNameRu() + "\n" +
                "Название (en): " + category.getNameEn() + "\n" +
                "Описание (ru): " + category.getDescriptionRu() + "\n" +
                "Описание (en): " + category.getDescriptionEn() + "\n" +
                "Текст кнопки (ru): " + category.getButtonTextRu() + "\n" +
                "Текст кнопки (en): " + category.getButtonTextEn() + "\n" +
                "Ссылка при нажатии на кнопку: " + category.getButtonLink() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage("Добавлена новая категория кешбэков", messageText);
        mailSender.send(message);
    }

    public void notifyCategoryUpdated(CashbackCategory category, User operator) {
        var messageText = "Категория изменена\n" +
                "ID: " + category.getId() + "\n" +
                "Название (ru): " + category.getNameRu() + "\n" +
                "Название (en): " + category.getNameEn() + "\n" +
                "Описание (ru): " + category.getDescriptionRu() + "\n" +
                "Описание (en): " + category.getDescriptionEn() + "\n" +
                "Текст кнопки (ru): " + category.getButtonTextRu() + "\n" +
                "Текст кнопки (en): " + category.getButtonTextEn() + "\n" +
                "Ссылка при нажатии на кнопку: " + category.getButtonLink() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage("Изменена категория кешбэков", messageText);
        mailSender.send(message);
    }

    public void notifyProgramCreated(CashbackProgram program, User operator) {
        var messageText = "Добавлена новая программа\n" +
                "ID: " + program.getId() + "\n" +
                "ID категории: " + program.getCategoryId() + "\n" +
                "Название категории: " + program.getCategoryNameRu() + "\n" +
                "Процент: " + toDisplayPercent(program.getPercent()) + "\n" +
                "Название (ru): " + program.getNameRu() + "\n" +
                "Название (en): " + program.getNameEn() + "\n" +
                "Текст в всплывающем окне (ru): " + program.getTooltipInfoRu() + "\n" +
                "Текст в всплывающем окне (en): " + program.getTooltipInfoEn() + "\n" +
                "Текст ссылки в всплывающем окне (ru): " + program.getTooltipLinkTextRu() + "\n" +
                "Текст ссылки в всплывающем окне (en): " + program.getTooltipLinkTextEn() + "\n" +
                "Ссылка в всплывающем окне: " + program.getTooltipLink() + "\n" +
                "Публичная: " + program.getIsPublic() + "\n" +
                "Активная: " + program.getIsEnabled() + "\n" +
                "Техническая: " + program.getIsTechnical() + "\n" +
                "Новая: " + program.getIsNew() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage("Добавлена новая программа кешбэков", messageText);
        mailSender.send(message);
    }

    public void notifyProgramUpdated(CashbackProgram program, User operator) {
        var messageText = "Программа изменена\n" +
                "ID: " + program.getId() + "\n" +
                "ID категории: " + program.getCategoryId() + "\n" +
                "Название категории: " + program.getCategoryNameRu() + "\n" +
                "Процент: " + toDisplayPercent(program.getPercent()) + "\n" +
                "Название (ru): " + program.getNameRu() + "\n" +
                "Название (en): " + program.getNameEn() + "\n" +
                "Текст в всплывающем окне (ru): " + program.getTooltipInfoRu() + "\n" +
                "Текст в всплывающем окне (en): " + program.getTooltipInfoEn() + "\n" +
                "Текст ссылки в всплывающем окне (ru): " + program.getTooltipLinkTextRu() + "\n" +
                "Текст ссылки в всплывающем окне (en): " + program.getTooltipLinkTextEn() + "\n" +
                "Ссылка в всплывающем окне: " + program.getTooltipLink() + "\n" +
                "Публичная: " + program.getIsPublic() + "\n" +
                "Активная: " + program.getIsEnabled() + "\n" +
                "Техническая: " + program.getIsTechnical() + "\n" +
                "Новая: " + program.getIsNew() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage("Изменена программа кешбэков", messageText);
        mailSender.send(message);
    }

    public void notifyProgramCategoryLinkCreated(CashbackProgramCategoryExtended link, User operator) {
        var subject = "Добавлена связь между программой и категорией кешбэков";
        var messageText = "Добавлена связь между программой и категорией\n" +
                "ID: " + link.getId() + "\n" +
                "ID категории: " + link.getCategoryId() + "\n" +
                "Название категории: " + link.getCategoryNameRu() + "\n" +
                "ID программы: " + link.getProgramId() + "\n" +
                "Название программы: " + link.getProgramNameRu() + "\n" +
                "Порядок программы (order): " + link.getOrder() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage(subject, messageText);
        mailSender.send(message);
    }

    public void notifyProgramCategoryLinkUpdated(CashbackProgramCategoryExtended link, User operator) {
        var subject = "Изменена связь между программой и категорией кешбэков";
        var messageText = "Изменена связь между программой и категорией\n" +
                "ID: " + link.getId() + "\n" +
                "ID категории: " + link.getCategoryId() + "\n" +
                "Название категории: " + link.getCategoryNameRu() + "\n" +
                "ID программы: " + link.getProgramId() + "\n" +
                "Название программы: " + link.getProgramNameRu() + "\n" +
                "Порядок программы (order): " + link.getOrder() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage(subject, messageText);
        mailSender.send(message);
    }

    public void notifyProgramCategoryLinkDeleted(CashbackProgramCategory deletedLink, User operator) {
        var subject = "Удалена связь между программой и категорией кешбэков";
        var messageText = "Удалена связь между программой и категорией\n" +
                "ID: " + deletedLink.getId() + "\n" +
                "ID категории: " + deletedLink.getCategoryId() + "\n" +
                "ID программы: " + deletedLink.getProgramId() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage(subject, messageText);
        mailSender.send(message);
    }

    public void notifyClientsAddedToProgram(CashbackProgram program, List<ClientId> clientIds, User operator) {
        var messageText = "В программу добавлены клиенты\n" +
                "ID: " + program.getId() + "\n" +
                "ID категории: " + program.getCategoryId() + "\n" +
                "Название категории: " + program.getCategoryNameRu() + "\n" +
                "Процент: " + toDisplayPercent(program.getPercent()) + "\n" +
                "Количество клиентов: " + clientIds.size() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage("В программу кешбэков добавлены клиенты", messageText);
        mailSender.send(message);
    }

    public void notifyClientsRemovedFromProgram(CashbackProgram program, List<ClientId> clientIds, User operator) {
        var messageText = "Из программы удалены клиенты\n" +
                "ID: " + program.getId() + "\n" +
                "ID категории: " + program.getCategoryId() + "\n" +
                "Название категории: " + program.getCategoryNameRu() + "\n" +
                "Процент: " + toDisplayPercent(program.getPercent()) + "\n" +
                "Количество клиентов: " + clientIds.size() + "\n" +
                "Пользователь: " + operator.getLogin();
        var message = new ActionNotificationMailMessage("Из программы кешбэков удалены клиенты", messageText);
        mailSender.send(message);
    }
}
