package ru.yandex.direct.internaltools.tools.cashback.service;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.cashback.model.CashbackProgramCategory;
import ru.yandex.direct.core.entity.cashback.model.CashbackProgramCategoryExtended;
import ru.yandex.direct.core.entity.cashback.repository.CashbackProgramsCategoriesRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.internaltools.tools.cashback.repository.CashbackProgramsCategoriesWriteRepository;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

@Service
public class CashbackProgramsCategoriesWriteService {
    private final CashbackProgramsCategoriesRepository readRepository;
    private final CashbackProgramsCategoriesWriteRepository writeRepository;
    private final CashbackNotificationsService notificationsService;

    @Autowired
    public CashbackProgramsCategoriesWriteService(CashbackProgramsCategoriesRepository readRepository,
                                                  CashbackProgramsCategoriesWriteRepository writeRepository,
                                                  CashbackNotificationsService notificationsService) {
        this.readRepository = readRepository;
        this.writeRepository = writeRepository;
        this.notificationsService = notificationsService;
    }

    @Nullable
    public CashbackProgramCategoryExtended createProgramCategoryLink(User operator, CashbackProgramCategory link) {
        writeRepository.add(List.of(link));
        var createdLink = readRepository.getExtendedByIds(link.getCategoryId(), link.getProgramId());
        if (Objects.nonNull(createdLink)) {
            notificationsService.notifyProgramCategoryLinkCreated(createdLink, operator);
        }
        return createdLink;
    }

    public CashbackProgramCategoryExtended updateProgramCategoryLink(User operator, CashbackProgramCategory link) {
        var existingLink = readRepository.getByIds(link.getCategoryId(), link.getProgramId());
        link.setId(existingLink.getId());
        var appliedChanges = List.of(toAppliedChanges(link, existingLink));
        writeRepository.update(appliedChanges);
        var updatedLink = readRepository.getExtendedByIds(link.getCategoryId(), link.getProgramId());
        notificationsService.notifyProgramCategoryLinkUpdated(updatedLink, operator);
        return updatedLink;
    }

    @Nullable
    public Long deleteProgramCategoryLink(User operator, CashbackProgramCategory link) {
        var toDelete = readRepository.getByIds(link.getCategoryId(), link.getProgramId());
        if (Objects.nonNull(toDelete)) {
            writeRepository.delete(toDelete);
            notificationsService.notifyProgramCategoryLinkDeleted(toDelete, operator);
            return toDelete.getId();
        }
        return null;
    }

    private AppliedChanges<CashbackProgramCategory> toAppliedChanges(CashbackProgramCategory update,
                                                                     CashbackProgramCategory existing) {
        return new ModelChanges<>(update.getId(), CashbackProgramCategory.class)
                .processNotNull(update.getOrder(), CashbackProgramCategory.ORDER)
                .applyTo(existing);
    }
}
