package ru.yandex.direct.internaltools.tools.cashback.service;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.cashback.model.CashbackProgram;
import ru.yandex.direct.core.entity.cashback.repository.CashbackProgramsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.internaltools.tools.cashback.repository.CashbackProgramsWriteRepository;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

@Service
public class CashbackProgramsWriteService {

    private final CashbackProgramsRepository readRepository;
    private final CashbackProgramsWriteRepository writeRepository;
    private final CashbackClientsWriteService clientsService;
    private final CashbackNotificationsService notificationsService;

    @Autowired
    public CashbackProgramsWriteService(CashbackProgramsRepository readRepository,
                                        CashbackProgramsWriteRepository writeRepository,
                                        CashbackClientsWriteService clientsService,
                                        CashbackNotificationsService notificationsService) {
        this.readRepository = readRepository;
        this.writeRepository = writeRepository;
        this.clientsService = clientsService;
        this.notificationsService = notificationsService;
    }

    @Nullable
    public CashbackProgram createProgram(User operator, CashbackProgram program) {
        var ids = writeRepository.add(List.of(program));
        if (program.getIsPublic() && program.getIsEnabled()) {
            clientsService.enablePublicProgram(program.withId(ids.get(0)));
        }
        var createdProgram = ids.isEmpty() ? null : readRepository.get(ids.get(0));
        if (createdProgram != null) {
            notificationsService.notifyProgramCreated(createdProgram, operator);
        }
        return createdProgram;
    }

    public CashbackProgram updateProgram(User operator, CashbackProgram program) {
        var existingProgram = readRepository.get(program.getId());
        if (Objects.nonNull(program.getCategoryId())) {
            //Изменяем категорию
            program.setCategoryId(program.getCategoryId());
        }
        var appliedChanges = List.of(toAppliedChanges(program, existingProgram));
        writeRepository.update(appliedChanges);
        appliedChanges.forEach(ac -> clientsService.handleProgramStatusChange(operator, ac));
        var updatedProgram = readRepository.get(program.getId());
        notificationsService.notifyProgramUpdated(updatedProgram, operator);
        return updatedProgram;
    }

    private AppliedChanges<CashbackProgram> toAppliedChanges(CashbackProgram update, CashbackProgram existing) {
        return new ModelChanges<>(update.getId(), CashbackProgram.class)
                .processNotNull(update.getPercent(), CashbackProgram.PERCENT)
                .processNotNull(update.getNameRu(), CashbackProgram.NAME_RU)
                .processNotNull(update.getNameEn(), CashbackProgram.NAME_EN)
                .processNotNull(update.getTooltipInfoRu(), CashbackProgram.TOOLTIP_INFO_RU)
                .processNotNull(update.getTooltipInfoEn(), CashbackProgram.TOOLTIP_INFO_EN)
                .processNotNull(update.getTooltipLinkTextRu(), CashbackProgram.TOOLTIP_LINK_TEXT_RU)
                .processNotNull(update.getTooltipLinkTextEn(), CashbackProgram.TOOLTIP_LINK_TEXT_EN)
                .processNotNull(update.getTooltipLink(), CashbackProgram.TOOLTIP_LINK)
                .process(update.getCategoryId(), CashbackProgram.CATEGORY_ID)
                .process(update.getIsEnabled(), CashbackProgram.IS_ENABLED)
                .process(update.getIsPublic(), CashbackProgram.IS_PUBLIC)
                .process(update.getIsTechnical(), CashbackProgram.IS_TECHNICAL)
                .process(update.getIsNew(), CashbackProgram.IS_NEW)
                .applyTo(existing);
    }
}
