package ru.yandex.direct.internaltools.tools.cashback.tool;

import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.cashback.model.CashbackCategory;
import ru.yandex.direct.core.entity.cashback.service.CashbackCategoriesService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.cashback.model.CashbackCategoriesParams;
import ru.yandex.direct.internaltools.tools.cashback.model.InternalToolsCashbackCategory;
import ru.yandex.direct.internaltools.tools.cashback.service.CashbackCategoriesWriteService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.CATEGORY_BUTTON_TEXT_MAX_LENGTH;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.CATEGORY_DESCRIPTION_MAX_LENGTH;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.CATEGORY_NAME_MAX_LENGTH;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.getIdFromCategoryKey;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.toCashbackCategory;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.toInternalToolsCashbackCategory;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.admissibleChars;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@Tool(
        name = "Управление категориями кешбэков",
        label = "cashback_categories",
        description = "Управление категориями кешбэков. По умолчанию выводится список всех существующих " +
                "категорий. Для создания категории надо заполнить все поля кроме \"Категория\". Если поле " +
                "\"Категория\" заполнено, то инструмент попытается обновить выбранную категорию.",
        consumes = CashbackCategoriesParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SEND)
@Category(InternalToolCategory.CASHBACK)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPERREADER})
@ParametersAreNonnullByDefault
public class CashbackCategoriesTool extends MassInternalTool<CashbackCategoriesParams, InternalToolsCashbackCategory> {

    @Autowired
    private CashbackCategoriesService cashbackCategoriesService;

    @Autowired
    private CashbackCategoriesWriteService cashbackCategoriesWriteService;

    /**
     * Возвращает все существующие категории кешбэков
     *
     * @return список всех существующих категорий кешбэков
     */
    @Nullable
    @Override
    protected List<InternalToolsCashbackCategory> getMassData() {
        var categories = cashbackCategoriesService.getAllCategories();
        return mapList(categories, InternalToolsCashbackConverter::toInternalToolsCashbackCategory);
    }

    /**
     * Создаёт или обновляет категорию кешбэка.
     * <p>
     * Если в параметрах не указана {@link CashbackCategoriesParams#getCategoryKey()},
     * то будет предпринята попытка создать новую категорию, иначе попытаемся обновить не пустые поля категории.
     *
     * @return Список из одной единственной категории — которую создали или обновили
     */
    @Override
    protected List<InternalToolsCashbackCategory> getMassData(CashbackCategoriesParams parameter) {
        var categoryId = getIdFromCategoryKey(parameter.getCategoryKey());
        var selectedCategory = cashbackCategoriesService.getCategoryById(categoryId);
        var category = toCashbackCategory(parameter);
        CashbackCategory result;
        if (selectedCategory != null) {
            category.withId(selectedCategory.getId());
            result = cashbackCategoriesWriteService.updateCategory(parameter.getOperator(), category);
        } else {
            result = cashbackCategoriesWriteService.createCategory(parameter.getOperator(), category);
        }
        return result == null ? List.of() : List.of(toInternalToolsCashbackCategory(result));
    }

    @Override
    public ValidationResult<CashbackCategoriesParams, Defect> validate(CashbackCategoriesParams params) {
        var isButtonFilled = Objects.nonNull(params.getButtonTextRu()) ||
                Objects.nonNull(params.getButtonTextEn()) || Objects.nonNull(params.getButtonLink());

        ItemValidationBuilder<CashbackCategoriesParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);

        var categoryKey = params.getCategoryKey();
        var creationModeOn = categoryKey == null || categoryKey.isEmpty();
        vb.item(params.getOperator(), "user")
                .checkBy(InternalToolsCashbackConstants.WRITE_PERMISSION_VALIDATOR, When.isTrue(creationModeOn));
        vb.item(params.getNameRu(), "nameRu")
                .check(notNull(), When.isTrue(creationModeOn))
                .check(notBlank(), When.isTrue(creationModeOn))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(CATEGORY_NAME_MAX_LENGTH), When.notNull());
        vb.item(params.getNameEn(), "nameEn")
                .check(notNull(), When.isTrue(creationModeOn))
                .check(notBlank(), When.isTrue(creationModeOn))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(CATEGORY_NAME_MAX_LENGTH), When.notNull());
        vb.item(params.getDescriptionRu(), "descriptionRu")
                .check(notNull(), When.isTrue(creationModeOn))
                .check(notBlank(), When.isTrue(creationModeOn))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(CATEGORY_DESCRIPTION_MAX_LENGTH), When.notNull());
        vb.item(params.getDescriptionEn(), "descriptionEn")
                .check(notNull(), When.isTrue(creationModeOn))
                .check(notBlank(), When.isTrue(creationModeOn))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(CATEGORY_DESCRIPTION_MAX_LENGTH), When.notNull());
        vb.item(params.getButtonTextRu(), "buttonTextRu")
                .check(notNull(), When.isTrue(creationModeOn && isButtonFilled))
                .check(notBlank(), When.isTrue(creationModeOn && isButtonFilled))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(CATEGORY_BUTTON_TEXT_MAX_LENGTH), When.notNull());
        vb.item(params.getButtonTextEn(), "buttonTextEn")
                .check(notNull(), When.isTrue(creationModeOn && isButtonFilled))
                .check(notBlank(), When.isTrue(creationModeOn && isButtonFilled))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(CATEGORY_BUTTON_TEXT_MAX_LENGTH), When.notNull());
        vb.item(params.getButtonLink(), "buttonLink")
                .check(notNull(), When.isTrue(creationModeOn && isButtonFilled))
                .check(notBlank(), When.isTrue(creationModeOn && isButtonFilled))
                .check(validHref(), When.notNull());
        if (Objects.nonNull(categoryKey)) {
            var categoryId = getIdFromCategoryKey(categoryKey);
            var category = cashbackCategoriesService.getCategoryById(categoryId);
            vb.item(category, "category")
                    .check(notNull());
        }

        return vb.getResult();
    }
}
