package ru.yandex.direct.internaltools.tools.cashback.tool;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.cashback.service.CashbackProgramsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.cashback.model.CashbackClientManageResponse;
import ru.yandex.direct.internaltools.tools.cashback.model.CashbackProgramKey;
import ru.yandex.direct.internaltools.tools.cashback.model.ManageClientsCashbackParams;
import ru.yandex.direct.internaltools.tools.cashback.service.CashbackClientsWriteService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.constraints.Constraints.validClientId;
import static ru.yandex.direct.internaltools.tools.cashback.model.ManageClientsCashbackParams.Action.ADD;
import static ru.yandex.direct.internaltools.tools.cashback.model.ManageClientsCashbackParams.Action.REMOVE;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.WRITE_PERMISSION_VALIDATOR;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Tool(
        name = "Управление программами кешбэка клиентов",
        label = "manage_cashback_clients",
        description = "Добавить или удалить клиентов из программы кешбэка",
        consumes = ManageClientsCashbackParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.CASHBACK)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPERREADER,
        InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class CashbackClientsManageTool extends MassInternalTool<ManageClientsCashbackParams, CashbackClientManageResponse> {

    @Autowired
    private CashbackProgramsService cashbackProgramsService;

    @Autowired
    private CashbackClientsWriteService cashbackClientsWriteService;

    @SuppressWarnings("ConstantConditions")
    @Override
    protected List<CashbackClientManageResponse> getMassData(ManageClientsCashbackParams params) {
        if (Objects.isNull(params.getClientIds()) && Objects.isNull(params.getFile())) {
            return List.of(new CashbackClientManageResponse().withMessage("Необходимо указать список ClientID или загрузить " +
                    "файл со списком"));
        }

        var clientIds = collectClientIds(params);
        var programId = CashbackProgramKey.parseId(params.getProgramKey());
        var program = cashbackProgramsService.getProgramById(programId);
        if (params.getAction() == ADD) {
            cashbackClientsWriteService.addClientsToProgram(params.getOperator(), program, clientIds);
            return List.of(new CashbackClientManageResponse().withMessage("Клиенты добавлены в программу"));
        } else if (params.getAction() == REMOVE) {
            cashbackClientsWriteService.removeClientsFromProgram(params.getOperator(), program, clientIds);
            return List.of(new CashbackClientManageResponse().withMessage("Клиенты удалены из программы"));
        }
        return List.of(new CashbackClientManageResponse().withMessage("Неизвестно действие"));
    }

    @Override
    public ValidationResult<ManageClientsCashbackParams, Defect> validate(ManageClientsCashbackParams params) {
        if (Objects.isNull(params.getClientIds()) && Objects.isNull(params.getFile())) {
            return ValidationResult.success(params); // Эта ситуация будет обработана в getMassData
        }
        ItemValidationBuilder<ManageClientsCashbackParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);

        var clientIds = collectClientIds(params);
        vb.list(clientIds, "clientIds")
                .check(notEmptyCollection())
                .checkEach(validClientId());

        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        vb.item(params.getOperator(), "user")
                .checkBy(WRITE_PERMISSION_VALIDATOR, When.isTrue(clientIds.size() > 1));

        vb.item(params.getProgramKey(), "program key")
                .check(notNull());

        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        var programId = CashbackProgramKey.parseId(params.getProgramKey());
        var program = cashbackProgramsService.getProgramById(programId);
        vb.item(program, "program")
                .check(notNull());
        return vb.getResult();
    }

    private List<ClientId> collectClientIds(ManageClientsCashbackParams params) {
        List<Long> clientIds = new ArrayList<>();
        if (params.getClientIds() != null) {
            clientIds.addAll(getLongIdsFromString(params.getClientIds()));
        }
        if (params.getFile() != null) {
            clientIds.addAll(getLongIdsFromString(new String(params.getFile())));
        }
        return mapList(clientIds, ClientId::fromLong);
    }
}
