package ru.yandex.direct.internaltools.tools.cashback.tool;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.cashback.model.CashbackProgramCategory;
import ru.yandex.direct.core.entity.cashback.service.CashbackCategoriesService;
import ru.yandex.direct.core.entity.cashback.service.CashbackProgramsCategoriesService;
import ru.yandex.direct.core.entity.cashback.service.CashbackProgramsService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.cashback.model.CashbackProgramCategoryParams;
import ru.yandex.direct.internaltools.tools.cashback.model.CashbackProgramKey;
import ru.yandex.direct.internaltools.tools.cashback.model.InternalToolsCashbackProgramCategory;
import ru.yandex.direct.internaltools.tools.cashback.service.CashbackProgramsCategoriesWriteService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.WRITE_PERMISSION_VALIDATOR;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.getIdFromCategoryKey;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.toInternalToolsCashbackProgramCategory;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;


@Tool(
        name = "Управление связью категорий и программ кешбеков",
        label = "cashback_program_category",
        description = "Управление связью между программами и категориями кешбеков. " +
                "В режиме редактирования изменяемым является порядок программы в карточке категории.",
        consumes = CashbackProgramCategoryParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.CASHBACK)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPERREADER})
@ParametersAreNonnullByDefault
@SuppressWarnings("checkstyle:LineLength")
public class CashbackProgramCategoryTool extends MassInternalTool<CashbackProgramCategoryParams, InternalToolsCashbackProgramCategory> {
    @Autowired
    private CashbackProgramsCategoriesService cashbackProgramsCategoriesService;
    @Autowired
    private CashbackProgramsCategoriesWriteService cashbackProgramsCategoriesWriteService;
    @Autowired
    private CashbackCategoriesService cashbackCategoriesService;
    @Autowired
    private CashbackProgramsService cashbackProgramsService;


    /**
     * Возвращает все связи между программами и категориями кешбеков, аггрегируя их по категориям
     * @return все связи программ и категорий кешбека
     */
    @Nullable
    @Override
    protected List<InternalToolsCashbackProgramCategory> getMassData() {
        var data = cashbackProgramsCategoriesService.getAllExtended();
        return mapList(data, InternalToolsCashbackConverter::toInternalToolsCashbackProgramCategory);
    }

    @Override
    protected List<InternalToolsCashbackProgramCategory> getMassData(CashbackProgramCategoryParams params) {
        CashbackProgramCategory programCategory;
        switch (params.getAction()) {
            case CREATE:
                programCategory = fromParams(params);
                var createdProgramCategory = cashbackProgramsCategoriesWriteService.createProgramCategoryLink(params.getOperator(), programCategory);
                return Objects.isNull(createdProgramCategory) ? List.of() : List.of(toInternalToolsCashbackProgramCategory(createdProgramCategory));
            case UPDATE:
                programCategory = fromParams(params);
                var updatedProgramCategory = cashbackProgramsCategoriesWriteService.updateProgramCategoryLink(params.getOperator(), programCategory);
                return List.of(toInternalToolsCashbackProgramCategory(updatedProgramCategory));
            case DELETE:
                var programId = CashbackProgramKey.parseId(params.getProgramKey());
                var categoryId = getIdFromCategoryKey(params.getCategoryKey());
                programCategory = new CashbackProgramCategory().withCategoryId(categoryId).withProgramId(programId);
                cashbackProgramsCategoriesWriteService.deleteProgramCategoryLink(params.getOperator(), programCategory);
                return getMassData();
            case SHOW:
            default:
                var data = cashbackProgramsCategoriesService.getAllExtended();
                if (Objects.nonNull(params.getCategoryKey())) {
                    var selectedCategoryId = getIdFromCategoryKey(params.getCategoryKey());
                    data = StreamEx.of(data).filter(o -> o.getCategoryId().equals(selectedCategoryId)).toList();
                }
                if (Objects.nonNull(params.getProgramKey())) {
                    var selectedProgramId = CashbackProgramKey.parseId(params.getProgramKey());
                    data = StreamEx.of(data).filter(o -> o.getProgramId().equals(selectedProgramId)).toList();
                }
                return mapList(data, InternalToolsCashbackConverter::toInternalToolsCashbackProgramCategory);
        }
    }

    @Override
    public ValidationResult<CashbackProgramCategoryParams, Defect> validate(CashbackProgramCategoryParams params) {
        switch (params.getAction()) {
            case CREATE:
                return validateCreate(params);
            case UPDATE:
                return validateUpdate(params);
            case DELETE:
                return validateDelete(params);
            case SHOW:
            default:
                return validateShow(params);
        }
    }

    private ValidationResult<CashbackProgramCategoryParams, Defect> validateCreate(CashbackProgramCategoryParams params) {
        ItemValidationBuilder<CashbackProgramCategoryParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        validateParamsNotNull(vb, params);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        var category = cashbackCategoriesService.getCategoryById(getIdFromCategoryKey(params.getCategoryKey()));
        vb.item(params.getCategoryKey(), "category")
                .check(unconditional(invalidValue()), When.isTrue(Objects.isNull(category)));
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        validateProgram(vb, params);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        validateOrder(vb, params);
        vb.item(params.getCategoryKey(), "category").check(unconditional(invalidValue()),
                When.isTrue(isLinkExists(category.getId(), CashbackProgramKey.parseId(params.getProgramKey()))));
        return vb.getResult();
    }

    private ValidationResult<CashbackProgramCategoryParams, Defect> validateUpdate(CashbackProgramCategoryParams params) {
        ItemValidationBuilder<CashbackProgramCategoryParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        validateParamsNotNull(vb, params);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        var category = cashbackCategoriesService.getCategoryById(getIdFromCategoryKey(params.getCategoryKey()));
        vb.item(params.getCategoryKey(), "category")
                .check(unconditional(invalidValue()), When.isTrue(Objects.isNull(category)));
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        validateProgram(vb, params);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        validateOrder(vb, params);
        vb.item(params.getCategoryKey(), "category").check(unconditional(invalidValue()),
                When.isFalse(isLinkExists(category.getId(), CashbackProgramKey.parseId(params.getProgramKey()))));
        return vb.getResult();
    }

    private ValidationResult<CashbackProgramCategoryParams, Defect> validateDelete(CashbackProgramCategoryParams params) {
        ItemValidationBuilder<CashbackProgramCategoryParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getOperator(), "user")
                .checkBy(WRITE_PERMISSION_VALIDATOR);
        vb.item(params.getCategoryKey(), "category").check(notNull());
        vb.item(params.getProgramKey(), "program").check(notNull());
        vb.item(params.getOrder(), "order").check(isNull());
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        var category = cashbackCategoriesService.getCategoryById(getIdFromCategoryKey(params.getCategoryKey()));
        vb.item(params.getCategoryKey(), "category")
                .check(unconditional(invalidValue()), When.isTrue(Objects.isNull(category)));
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        validateProgram(vb, params);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        vb.item(params.getCategoryKey(), "category").check(unconditional(invalidValue()),
                When.isFalse(isLinkExists(category.getId(), CashbackProgramKey.parseId(params.getProgramKey()))));
        return vb.getResult();
    }

    private ValidationResult<CashbackProgramCategoryParams, Defect> validateShow(CashbackProgramCategoryParams params) {
        ItemValidationBuilder<CashbackProgramCategoryParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getOrder(), "order").check(isNull());
        if (Objects.nonNull(params.getCategoryKey())) {
            var category = cashbackCategoriesService.getCategoryById(getIdFromCategoryKey(params.getCategoryKey()));
            vb.item(params.getCategoryKey(), "category")
                    .check(unconditional(invalidValue()), When.isTrue(Objects.isNull(category)));
        }
        if (Objects.nonNull(params.getProgramKey())) {
            validateProgram(vb, params);
        }
        return vb.getResult();
    }

    private CashbackProgramCategory fromParams(CashbackProgramCategoryParams params) {
        return new CashbackProgramCategory()
                .withCategoryId(getIdFromCategoryKey(params.getCategoryKey()))
                .withProgramId(CashbackProgramKey.parseId(params.getProgramKey()))
                .withOrder(Long.parseLong(params.getOrder()));
    }

    private void validateParamsNotNull(ItemValidationBuilder<CashbackProgramCategoryParams, Defect> vb,
                                       CashbackProgramCategoryParams params) {
        vb.item(params.getOperator(), "user")
                .checkBy(WRITE_PERMISSION_VALIDATOR);
        vb.item(params.getCategoryKey(), "category").check(notNull());
        vb.item(params.getProgramKey(), "program").check(notNull());
        vb.item(params.getOrder(), "order").check(notNull());
    }

    private void validateOrder(ItemValidationBuilder<CashbackProgramCategoryParams, Defect> vb,
                               CashbackProgramCategoryParams params) {
        try {
            var order = Long.parseLong(params.getOrder());
        } catch (NumberFormatException ignore) {
            vb.item(params.getOrder(), "order")
                    .check(unconditional(invalidValue()));
        }
    }

    private void validateProgram(ItemValidationBuilder<CashbackProgramCategoryParams, Defect> vb,
                                 CashbackProgramCategoryParams params) {
        var programId = CashbackProgramKey.parseId(params.getProgramKey());
        var program = cashbackProgramsService.getProgramById(programId);
        vb.item(params.getProgramKey(), "program")
                .check(unconditional(invalidValue()), When.isTrue(Objects.isNull(programId)))
                .check(unconditional(invalidValue()), When.isTrue(Objects.isNull(program)));
    }

    private boolean isLinkExists(Long categoryId, Long programId) {
        return !StreamEx.of(cashbackProgramsCategoriesService.getAllExtended())
                .filter(o -> o.getCategoryId().equals(categoryId) && o.getProgramId().equals(programId))
                .toList().isEmpty();
    }
}
