package ru.yandex.direct.internaltools.tools.cashback.tool;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.cashback.model.CashbackProgram;
import ru.yandex.direct.core.entity.cashback.service.CashbackCategoriesService;
import ru.yandex.direct.core.entity.cashback.service.CashbackProgramsService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.cashback.model.CashbackProgramKey;
import ru.yandex.direct.internaltools.tools.cashback.model.CashbackProgramsParams;
import ru.yandex.direct.internaltools.tools.cashback.model.InternalToolsCashbackProgram;
import ru.yandex.direct.internaltools.tools.cashback.service.CashbackProgramsWriteService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.cashback.CashbackConstants.TECHNICAL_PROGRAM_ID;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.PERCENT_MAX_VALUE;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.PERCENT_MAX_VALUE_INT;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.PERCENT_MIN_VALUE;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.PROGRAM_NAME_MAX_LENGTH;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.PROGRAM_TOOLTIP_INFO_MAX_LENGTH;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.PROGRAM_TOOLTIP_LINK_TEXT_MAX_LENGTH;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConstants.WRITE_PERMISSION_VALIDATOR;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.getIdFromCategoryKey;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.toCashbackProgram;
import static ru.yandex.direct.internaltools.tools.cashback.tool.InternalToolsCashbackConverter.toInternalToolsCashbackProgram;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.StringConstraints.admissibleChars;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.mustBeEmpty;

@Tool(
        name = "Управление программами кешбэков",
        label = "cashback_programs",
        description = "Управление программами кешбэков. Флаги включенности и публичности в режимах редактирования и " +
                "создания программы являются обязательными. " +
                "Максимальное значение процента программы - " + PERCENT_MAX_VALUE_INT,
        consumes = CashbackProgramsParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.CASHBACK)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPERREADER})
@ParametersAreNonnullByDefault
public class CashbackProgramsTool extends MassInternalTool<CashbackProgramsParams, InternalToolsCashbackProgram> {
    @Autowired
    private CashbackProgramsService cashbackProgramsService;

    @Autowired
    private CashbackProgramsWriteService cashbackProgramsWriteService;

    @Autowired
    private CashbackCategoriesService cashbackCategoriesService;

    /**
     * Возвращает все существующие программы кешбэка.
     *
     * @return Все программы кешбэка
     */
    @Nullable
    @Override
    protected List<InternalToolsCashbackProgram> getMassData() {
        var programs = cashbackProgramsService.getAllPrograms();
        return mapList(programs, InternalToolsCashbackConverter::toInternalToolsCashbackProgram);
    }

    @Override
    protected List<InternalToolsCashbackProgram> getMassData(CashbackProgramsParams params) {
        var categoryId = getIdFromCategoryKey(params.getCategoryKey());
        CashbackProgram program;
        switch (params.getAction()) {
            case CREATE:
                var selectedCategory = cashbackCategoriesService.getCategoryById(categoryId);
                program = toCashbackProgram(params);
                if (Objects.nonNull(selectedCategory)) {
                    program = program.withCategoryId(selectedCategory.getId());
                }
                var createdProgram =
                        cashbackProgramsWriteService.createProgram(params.getOperator(), program);
                return createdProgram == null ? List.of() : List.of(toInternalToolsCashbackProgram(createdProgram));
            case UPDATE:
                program = toCashbackProgram(params);
                var selectedProgramId = CashbackProgramKey.parseId(params.getProgramKey());
                program.withId(selectedProgramId);
                program.withCategoryId(categoryId);
                var updatedProgram = cashbackProgramsWriteService.updateProgram(params.getOperator(), program);
                return List.of(toInternalToolsCashbackProgram(updatedProgram));
            case SHOW:
            default:
                if (categoryId == null) {
                    return getMassData();
                }

                var programsInCategory = cashbackProgramsService.getProgramsByCategory(categoryId);
                return mapList(programsInCategory, InternalToolsCashbackConverter::toInternalToolsCashbackProgram);
        }
    }

    @Override
    public ValidationResult<CashbackProgramsParams, Defect> validate(CashbackProgramsParams params) {
        switch (params.getAction()) {
            case CREATE:
                return validateCreate(params);
            case UPDATE:
                return validateUpdate(params);
            case SHOW:
            default:
                return validateShow(params);
        }
    }

    private ValidationResult<CashbackProgramsParams, Defect> validateShow(CashbackProgramsParams params) {
        ItemValidationBuilder<CashbackProgramsParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        var category = cashbackCategoriesService.getCategoryById(getIdFromCategoryKey(params.getCategoryKey()));
        vb.item(params.getCategoryKey(), "category")
                .check(unconditional(invalidValue()), When.isTrue(params.getCategoryKey() != null && category == null));
        vb.item(params.getProgramKey(), "program").check(isNull(), mustBeEmpty());
        vb.item(params.getPercent(), "percent").check(isNull(), mustBeEmpty());
        return vb.getResult();
    }

    private ValidationResult<CashbackProgramsParams, Defect> validateCreate(CashbackProgramsParams params) {
        ItemValidationBuilder<CashbackProgramsParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);

        vb.item(params.getOperator(), "user")
                .checkBy(WRITE_PERMISSION_VALIDATOR);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        if (Objects.nonNull(params.getCategoryKey())) {
            var category = cashbackCategoriesService.getCategoryById(getIdFromCategoryKey(params.getCategoryKey()));
            vb.item(category, "category")
                    .check(notNull());
            if (vb.getResult().hasAnyErrors()) {
                return vb.getResult();
            }
        }

        vb.item(params.getProgramKey(), "program")
                .check(isNull());
        vb.item(params.getPercent(), "percent")
                .check(notNull());
        vb.item(params.getNameRu(), "name_ru")
                .check(notNull())
                .check(notBlank())
                .check(admissibleChars())
                .check(maxStringLength(PROGRAM_NAME_MAX_LENGTH));
        vb.item(params.getNameEn(), "name_en")
                .check(notNull())
                .check(notBlank())
                .check(admissibleChars())
                .check(maxStringLength(PROGRAM_NAME_MAX_LENGTH));
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        validatePercent(vb, params);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        validateTooltip(vb, params);
        return vb.getResult();
    }

    private ValidationResult<CashbackProgramsParams, Defect> validateUpdate(CashbackProgramsParams params) {
        ItemValidationBuilder<CashbackProgramsParams, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getOperator(), "user")
                .checkBy(WRITE_PERMISSION_VALIDATOR);
        vb.item(params.getProgramKey(), "program key")
                .check(notNull());
        if (Objects.nonNull(params.getPercent())) {
            validatePercent(vb, params);
        }
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        var selectedProgramId = CashbackProgramKey.parseId(params.getProgramKey());
        vb.item(selectedProgramId, "program id")
                .check(notNull())
                .check(notEqual(TECHNICAL_PROGRAM_ID, invalidValue())); // Технические программы запрещаем редактировать

        var program = cashbackProgramsService.getProgramById(selectedProgramId);
        vb.item(program, "program")
                .check(notNull());
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        var isNowPublic = !program.getIsPublic() && params.getIsPublic();
        vb.item(params.getIsEnabled(), "isEnabled")
                .check(isEqual(false, invalidValue()), When.isTrue(isNowPublic));
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        vb.item(params.getNameRu(), "name_ru")
                .check(notBlank(), When.notNull())
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(PROGRAM_NAME_MAX_LENGTH), When.notNull());
        vb.item(params.getNameEn(), "name_en")
                .check(notBlank(), When.notNull())
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(PROGRAM_NAME_MAX_LENGTH), When.notNull());
        validateTooltip(vb, params);
        return vb.getResult();
    }

    private void validateTooltip(ItemValidationBuilder<CashbackProgramsParams, Defect> vb,
                                 CashbackProgramsParams params) {
        boolean isTooltipSet = isTooltipSet(params);
        vb.item(params.getTooltipInfoRu(), "tooltip_info_ru")
                .check(notNull(), When.isTrue(isTooltipSet))
                .check(notBlank(), When.isTrue(isTooltipSet))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(PROGRAM_TOOLTIP_INFO_MAX_LENGTH), When.notNull());
        vb.item(params.getTooltipInfoEn(), "tooltip_info_en")
                .check(notNull(), When.isTrue(isTooltipSet))
                .check(notBlank(), When.isTrue(isTooltipSet))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(PROGRAM_TOOLTIP_INFO_MAX_LENGTH), When.notNull());
        vb.item(params.getTooltipLinkTextRu(), "tooltip_link_text_ru")
                .check(notNull(), When.isTrue(isTooltipSet))
                .check(notBlank(), When.isTrue(isTooltipSet))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(PROGRAM_TOOLTIP_LINK_TEXT_MAX_LENGTH), When.notNull());
        vb.item(params.getTooltipLinkTextEn(), "tooltip_link_text_en")
                .check(notNull(), When.isTrue(isTooltipSet))
                .check(notBlank(), When.isTrue(isTooltipSet))
                .check(admissibleChars(), When.notNull())
                .check(maxStringLength(PROGRAM_TOOLTIP_LINK_TEXT_MAX_LENGTH), When.notNull());
        vb.item(params.getTooltipLink(), "tooltip_link")
                .check(notNull(), When.isTrue(isTooltipSet))
                .check(notBlank(), When.isTrue(isTooltipSet))
                .check(validHref(), When.notNull());
    }

    private void validatePercent(ItemValidationBuilder<CashbackProgramsParams, Defect> vb,
                                 CashbackProgramsParams params) {
        try {
            var percentValue = new BigDecimal(params.getPercent());
            vb.item(percentValue, "percent value")
                    .check(inRange(PERCENT_MIN_VALUE, PERCENT_MAX_VALUE));
        } catch (NumberFormatException ignore) {
            vb.item(params.getPercent(), "percent value")
                    .check(unconditional(invalidValue()));
        }
    }

    private static boolean isTooltipSet(CashbackProgramsParams params) {
        return Objects.nonNull(params.getTooltipInfoRu()) ||
                Objects.nonNull(params.getTooltipInfoEn()) ||
                Objects.nonNull(params.getTooltipLinkTextRu()) ||
                Objects.nonNull(params.getTooltipLinkTextEn()) ||
                Objects.nonNull(params.getTooltipLink());
    }
}
