package ru.yandex.direct.internaltools.tools.clientcountrycurrency;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.tools.clientcountrycurrency.container.ClientCountryAndCurrencies;

import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_FIRM_COUNTRY_CURRENCY;
import static ru.yandex.direct.dbschema.ppc.Tables.GEO_REGIONS;

@Repository
@ParametersAreNonnullByDefault
public class ClientCountryCurrencyRepository {
    private static final String COUNTRY_ALIAS = "country";
    private static final String CURRENCIES_ALIAS = "currencies";
    private static final String LAST_UPDATE_ALIAS = "last_update";

    private final DslContextProvider dslContextProvider;

    @Autowired
    public ClientCountryCurrencyRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }


    /**
     * Возвращает список триплетов 'страна-валюты-дата обновления' на клиента.
     */
    public List<ClientCountryAndCurrencies> getClientCountryAndCurrencies(int shard, Long clientId) {
        return dslContextProvider.ppc(shard)
                .select(GEO_REGIONS.NAME.as(COUNTRY_ALIAS),
                        DSL.groupConcatDistinct(CLIENT_FIRM_COUNTRY_CURRENCY.CURRENCY).separator(",")
                                .as(CURRENCIES_ALIAS),
                        DSL.max(CLIENT_FIRM_COUNTRY_CURRENCY.LAST_UPDATE).as(LAST_UPDATE_ALIAS)
                )
                .from(CLIENT_FIRM_COUNTRY_CURRENCY)
                .leftJoin(GEO_REGIONS).on(CLIENT_FIRM_COUNTRY_CURRENCY.COUNTRY_REGION_ID.eq(GEO_REGIONS.REGION_ID))
                .where(CLIENT_FIRM_COUNTRY_CURRENCY.CLIENT_ID.eq(clientId))
                .groupBy(CLIENT_FIRM_COUNTRY_CURRENCY.COUNTRY_REGION_ID)
                .fetch()
                .map(r -> new ClientCountryAndCurrencies(
                        r.get(COUNTRY_ALIAS, String.class),
                        r.get(CURRENCIES_ALIAS, String.class),
                        r.get(LAST_UPDATE_ALIAS, LocalDateTime.class)
                ));
    }
}
