package ru.yandex.direct.internaltools.tools.clientndsschedule;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.clientndsschedule.container.ClientNdsScheduleInfo;
import ru.yandex.direct.internaltools.tools.clientndsschedule.model.ClientNdsScheduleParameters;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;

@Tool(
        name = "График НДС клиента",
        label = "client_nds_schedule",
        description = "Просмотр графика НДС клиента",
        consumes = ClientNdsScheduleParameters.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.OTHER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER,
        InternalToolAccessRole.MANAGER, InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class ClientNdsScheduleTool extends MassInternalTool<ClientNdsScheduleParameters, ClientNdsScheduleInfo> {
    private ClientNdsService clientNdsService;
    private UserService userService;
    private ShardHelper shardHelper;

    @Autowired
    public ClientNdsScheduleTool(ClientNdsService clientNdsService,
                                 UserService userService, ShardHelper shardHelper) {
        this.clientNdsService = clientNdsService;
        this.userService = userService;
        this.shardHelper = shardHelper;
    }

    @Override
    protected List<ClientNdsScheduleInfo> getMassData(ClientNdsScheduleParameters parameter) {
        String login = PassportUtils.normalizeLogin(parameter.getLogin());
        ClientId clientId = ClientId.fromLong(shardHelper.getClientIdByLogin(login));
        Collection<ClientNds> clientNdsHistory = clientNdsService.getClientNdsHistory(clientId);
        return clientNdsHistory == null ? Collections.emptyList()
                : clientNdsHistory.stream().map(ClientNdsScheduleInfo::new).collect(Collectors.toList());
    }

    @Override
    public ValidationResult<ClientNdsScheduleParameters, Defect> validate(ClientNdsScheduleParameters param) {
        ItemValidationBuilder<ClientNdsScheduleParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(param);

        validationBuilder
                .item(PassportUtils.normalizeLogin(param.getLogin()), "Login: ")
                .check(notNull())
                .check(notEmpty())
                .check(loginExists());

        return validationBuilder.getResult();
    }

    private Constraint<String, Defect> loginExists() {
        return s -> userService.loginExists(s) ? null : CommonDefects.objectNotFound();
    }

}
