package ru.yandex.direct.internaltools.tools.communication;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.communication.model.CommunicationEvent;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.tools.communication.model.BaseCommunicationEventVersionsParameters;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventVersionsAction;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventVersionsResponse;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytwrapper.client.YtProvider;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.internaltools.tools.communication.util.SendPersonalCommunicationEventUtils.getEventIdValidator;
import static ru.yandex.direct.internaltools.tools.communication.util.SendPersonalCommunicationEventUtils.getEventIterationValidator;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;

abstract class BaseCommunicationEventVersionsTool<T extends BaseCommunicationEventVersionsParameters>
        implements BaseInternalTool<T> {
    protected static final Set<String> APPROVERS = Set.of(
            "yndx-a-dubov",
            "yndx-r-shakirova-developer"
    );

    protected Map<CommunicationEventVersionsAction, Validator<T, Defect>> validators;
    protected Map<CommunicationEventVersionsAction, Function<T, List<CommunicationEventVersion>>> executors;

    protected final CommunicationEventsRepository eventsRepository;
    protected final CommunicationEventVersionsRepository versionsRepository;
    protected final YtProvider ytProvider;
    protected final UserService userService;

    protected BaseCommunicationEventVersionsTool(CommunicationEventsRepository eventsRepository,
                                                 CommunicationEventVersionsRepository versionsRepository,
                                                 YtProvider ytProvider,
                                                 UserService userService) {
        this.eventsRepository = eventsRepository;
        this.versionsRepository = versionsRepository;
        this.ytProvider = ytProvider;
        this.userService = userService;

        validators = new HashMap<>();
        validators.put(CommunicationEventVersionsAction.SHOW, this::showValidation);
        validators.put(CommunicationEventVersionsAction.CREATE, this::createValidation);
        validators.put(CommunicationEventVersionsAction.COPY, this::copyValidation);
        validators.put(CommunicationEventVersionsAction.UPDATE, this::updateValidation);
        validators.put(CommunicationEventVersionsAction.REQUEST_APPROVE, getUserAndStatusValidator(false, Set.of(
                CommunicationEventVersionStatus.NEW_)));
        validators.put(CommunicationEventVersionsAction.APPROVE, getUserAndStatusValidator(true, Set.of(
                CommunicationEventVersionStatus.NEED_APPROVE)));
        validators.put(CommunicationEventVersionsAction.REJECT, getUserAndStatusValidator(false, Set.of(
                CommunicationEventVersionStatus.NEED_APPROVE,
                CommunicationEventVersionStatus.APPROVED,
                CommunicationEventVersionStatus.READY)));
        validators.put(CommunicationEventVersionsAction.START, getUserAndStatusValidator(false, Set.of(
                CommunicationEventVersionStatus.APPROVED)));
        validators.put(CommunicationEventVersionsAction.STOP, getUserAndStatusValidator(false, Set.of(
                CommunicationEventVersionStatus.ACTIVE,
                CommunicationEventVersionStatus.ACTIVATING)));
        validators.put(CommunicationEventVersionsAction.ARCHIVE, this::archiveValidation);

        executors = new HashMap<>();
        executors.put(CommunicationEventVersionsAction.SHOW, this::show);
        executors.put(CommunicationEventVersionsAction.CREATE, this::create);
        executors.put(CommunicationEventVersionsAction.COPY, this::copy);
        executors.put(CommunicationEventVersionsAction.UPDATE, this::update);
        executors.put(CommunicationEventVersionsAction.REQUEST_APPROVE, this::requestApprove);
        executors.put(CommunicationEventVersionsAction.APPROVE, this::approve);
        executors.put(CommunicationEventVersionsAction.REJECT, this::reject);
        executors.put(CommunicationEventVersionsAction.START, this::start);
        executors.put(CommunicationEventVersionsAction.STOP, this::stop);
        executors.put(CommunicationEventVersionsAction.ARCHIVE, this::archive);
    }

    protected abstract T makeParamsByVersionModel(CommunicationEventVersion cev);

    @Override
    public ValidationResult<T, Defect> validate(T params) {
        return validators.getOrDefault(params.getAction(), this::defaultValidation).apply(params);
    }

    @Override
    public InternalToolMassResult<CommunicationEventVersionsResponse> process(T params) {
        List<CommunicationEventVersion> versions = executors.get(params.getAction()).apply(params);
        List<CommunicationEventVersionsResponse> results = mapList(versions,
                v -> new CommunicationEventVersionsResponse(v));
        return new InternalToolMassResult(results);
    }

    protected ValidationResult<T, Defect> defaultValidation(T params) {
        return ItemValidationBuilder.<T, Defect>of(params).getResult();
    }

    protected ValidationResult<T, Defect> showValidation(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getEventId(), "event")
                .checkBy(getEventIdValidator(eventsRepository, List.of(CommunicationEventStatus.ARCHIVED)));
        return builder.getResult();
    }

    protected List<CommunicationEventVersion> archive(T params) {
        CommunicationEventVersion cev = versionsRepository.getVersion(params.getEventId(), params.getIteration());
        cev.setStatus(CommunicationEventVersionStatus.ARCHIVED);
        versionsRepository.update(cev, Set.of(
                CommunicationEventVersionStatus.NEW_,
                CommunicationEventVersionStatus.ABORTED,
                CommunicationEventVersionStatus.ACTIVE
        ));
        return List.of(versionsRepository.getVersion(params.getEventId(), params.getIteration()));
    }

    protected Validator<T, Defect> getUserAndStatusValidator(boolean isApprove,
                                                             Set<CommunicationEventVersionStatus> statuses) {
        return params -> {
            ItemValidationBuilder<T, Defect> builder =
                    ItemValidationBuilder.of(params);
            builder.checkBy(this::iterationValidation);
            var cev = versionsRepository
                    .getOptionalVersion(params.getEventId(), params.getIteration());
            if (cev.isEmpty()) {
                return builder.getResult();
            }

            if (isApprove) {
                builder.item(params.getOperator(), "rights")
                        .checkByFunction(this::validateApprover);
            } else {
                var ownerLogins =
                        StreamEx.of(eventsRepository.getCommunicationEventsByIds(List.of(params.getEventId())))
                                .findAny()
                                .map(CommunicationEvent::getOwners)
                                .orElse(emptySet());
                builder.item(params.getOperator().getLogin(), "rights")
                        .checkBy(getOwnerValidator(ownerLogins));
            }

            builder.item(cev.get().getStatus(), "action")
                    .check(inSet(statuses), inconsistentState());
            return builder.getResult();
        };
    }

    protected Defect validateApprover(User operator) {
        if (operator.getRole().equals(RbacRole.SUPER)) {
            return null;
        }
        if (APPROVERS.contains(operator.getLogin())) {
            return null;
        }
        return inconsistentState();
    }

    protected Validator<String, Defect> getOwnerValidator(Set<String> ownerLogins) {
        return user -> {
            ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(user);
            builder.check(inSet(ownerLogins));
            return builder.getResult();
        };
    }

    protected ValidationResult<T, Defect> archiveValidation(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        Set<CommunicationEventVersionStatus> validStatuses = new HashSet();
        validStatuses.add(CommunicationEventVersionStatus.NEW_);
        validStatuses.add(CommunicationEventVersionStatus.ABORTED);
        var cev = versionsRepository
                .getOptionalVersion(params.getEventId(), params.getIteration());
        if (cev.isPresent() && cev.get().getExpired().isBefore(LocalDateTime.now())) {
            validStatuses.add(CommunicationEventVersionStatus.ACTIVE);
        }
        builder.checkBy(getUserAndStatusValidator(false, validStatuses));
        return builder.getResult();
    }

    protected ValidationResult<T, Defect> iterationValidation(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getEventId(), "event")
                .checkBy(getEventIdValidator(eventsRepository, List.of(CommunicationEventStatus.ARCHIVED)));
        builder.item(params.getIteration(), "iteration")
                .checkBy(getEventIterationValidator(versionsRepository, params.getEventId()));
        return builder.getResult();
    }

    protected ItemValidationBuilder<T, Defect> createValidationBuilder(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getEventId(), "event")
                .checkBy(getEventIdValidator(eventsRepository, List.of(CommunicationEventStatus.ARCHIVED)));

        var ownerLogins = StreamEx.of(eventsRepository.getCommunicationEventsByIds(List.of(params.getEventId())))
                .findAny()
                .map(CommunicationEvent::getOwners)
                .orElse(emptySet());
        if (!ownerLogins.isEmpty()) {
            builder.item(params.getOperator().getLogin(), "rights")
                    .checkBy(getOwnerValidator(ownerLogins));
        }

        builder.checkBy(this::popupValidator);
        builder.checkBy(this::timeValidator);
        return builder;
    }

    protected ValidationResult<T, Defect> createValidation(T params) {
        var builder = createValidationBuilder(params);
        return builder.getResult();
    }

    protected ValidationResult<T, Defect> copyValidation(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getEventId(), "event")
                .checkBy(getEventIdValidator(eventsRepository, List.of(CommunicationEventStatus.ARCHIVED)));

        if (params.getIteration() == null) {
            params.withIteration(versionsRepository
                    .getVersionsByEvent(params.getEventId())
                    .stream()
                    .map(CommunicationEventVersion::getIter)
                    .max(Long::compareTo)
                    .orElse(null));
        }
        builder.item(params.getIteration(), "iteration")
                .checkBy(getEventIterationValidator(versionsRepository, params.getEventId()));

        var cev = versionsRepository
                .getOptionalVersion(params.getEventId(), params.getIteration());
        if (cev.isEmpty()) {
            return builder.getResult();
        }

        fillVersionModelByInput(cev.get(), params, true);
        var validateParams = makeParamsByVersionModel(cev.get());
        validateParams.setOperator(params.getOperator());
        builder = ItemValidationBuilder.of(validateParams);
        builder.checkBy(this::createValidation);

        return builder.getResult();
    }

    protected ValidationResult<T, Defect> updateValidation(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        builder.checkBy(this::iterationValidation);
        var cev = versionsRepository
                .getOptionalVersion(params.getEventId(), params.getIteration());
        if (cev.isEmpty()) {
            return builder.getResult();
        }

        fillVersionModelByInput(cev.get(), params, false);
        var validateParams = makeParamsByVersionModel(cev.get());
        validateParams.setOperator(params.getOperator());
        builder = ItemValidationBuilder.of(validateParams);
        builder.checkBy(this::createValidation);
        builder.checkBy(getUserAndStatusValidator(false, Set.of(CommunicationEventVersionStatus.NEW_)));
        return builder.getResult();
    }

    protected ValidationResult<T, Defect> popupValidator(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getTitle(), "title")
                .check(notNull());
        builder.item(params.getText(), "text")
                .check(notNull());
        builder.item(params, "button")
                .checkBy(this::buttonValidator);
        builder.item(params.getImageHref(), "image")
                .check(notNull())
                .check(validHref(), When.notNull());
        return builder.getResult();
    }

    protected ValidationResult<T, Defect> buttonValidator(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getButtonText(), "text")
                .check(notNull());
        builder.item(params.getButtonHref(), "href")
                .check(notNull())
                .check(validHref(), When.notNull());
        return builder.getResult();
    }

    protected ValidationResult<T, Defect> timeValidator(T params) {
        ItemValidationBuilder<T, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getStartEventTime(), "startTime")
                .check(notNull())
                .check(isNotBeforeThan(LocalDateTime.now()), When.isValid());
        if (params.getStartEventTime() == null) {
            return builder.getResult();
        }
        builder.item(params.getExpired(), "expired")
                .check(notNull(), When.isValid())
                .check(isNotBeforeThan(params.getStartEventTime()), When.isValid());
        return builder.getResult();
    }

    protected List<CommunicationEventVersion> show(T params) {
        var eventId = params.getEventId();
        var iter = params.getIteration();
        return iter == null ?
                versionsRepository.getVersionsByEvent(eventId) :
                List.of(versionsRepository.getVersion(eventId, iter));
    }

    protected List<CommunicationEventVersion> create(T params) {
        CommunicationEventVersion cev = new CommunicationEventVersion();
        fillVersionModelByInput(cev, params, true);
        cev.setStatus(CommunicationEventVersionStatus.NEW_);
        Long iter = versionsRepository
                .getVersionsByEvent(cev.getEventId())
                .stream()
                .map(CommunicationEventVersion::getIter)
                .max(Long::compareTo)
                .orElse(0L);
        cev.setIter(iter + 1);
        if (!versionsRepository.create(cev)) {
            return Collections.emptyList();
        }
        return List.of(versionsRepository.getVersion(cev.getEventId(), cev.getIter()));
    }

    protected List<CommunicationEventVersion> copy(T params) {
        Long iter = params.getIteration();
        if (iter == null) {
            params.withIteration(versionsRepository
                    .getVersionsByEvent(params.getEventId())
                    .stream()
                    .map(CommunicationEventVersion::getIter)
                    .max(Long::compareTo)
                    .orElse(null));
        }

        var cev = versionsRepository
                .getOptionalVersion(params.getEventId(), iter);
        fillVersionModelByInput(cev.get(), params, true);
        var createParams = makeParamsByVersionModel(cev.get());
        createParams.setOperator(params.getOperator());
        return create(createParams);
    }

    protected List<CommunicationEventVersion> update(T params) {
        var cev = versionsRepository
                .getVersion(params.getEventId(), params.getIteration());
        fillVersionModelByInput(cev, params, true);
        versionsRepository.update(cev, CommunicationEventVersionStatus.NEW_);
        return List.of(versionsRepository.getVersion(cev.getEventId(), cev.getIter()));
    }

    protected List<CommunicationEventVersion> requestApprove(T params) {
        CommunicationEventVersion cev = versionsRepository.getVersion(params.getEventId(), params.getIteration());
        cev.setStatus(CommunicationEventVersionStatus.NEED_APPROVE);
        // Если пользователи заданы не таблицей, а логинами
        if (cev.getUserTableHash() == null) {
            var logins = List.of(cev.getUsers().split(","));
            // Если все пользователи являются владельцами
            if (StreamEx.of(eventsRepository.getCommunicationEventsByIds(List.of(cev.getEventId())))
                    .findAny()
                    .map(CommunicationEvent::getOwners)
                    .filter(owners -> owners.containsAll(logins))
                    .isPresent()) {
                // Если все пользователи являются внутренними
                if (userService.massGetUserByLogin(logins)
                        .stream()
                        .map(User::getRole)
                        .allMatch(RbacRole::isInternal)) {
                    // Автоподтверждаем отправку попапа
                    cev.setStatus(CommunicationEventVersionStatus.APPROVED);
                }
            }

        }
        versionsRepository.update(cev, CommunicationEventVersionStatus.NEW_);
        return List.of(versionsRepository.getVersion(params.getEventId(), params.getIteration()));
    }

    protected List<CommunicationEventVersion> approve(T params) {
        CommunicationEventVersion cev = versionsRepository.getVersion(params.getEventId(), params.getIteration());
        cev.setStatus(CommunicationEventVersionStatus.APPROVED);
        versionsRepository.update(cev, CommunicationEventVersionStatus.NEED_APPROVE);
        return List.of(versionsRepository.getVersion(params.getEventId(), params.getIteration()));
    }

    protected List<CommunicationEventVersion> reject(T params) {
        CommunicationEventVersion cev = versionsRepository.getVersion(params.getEventId(), params.getIteration());
        cev.setStatus(CommunicationEventVersionStatus.NEW_);
        versionsRepository.update(cev, Set.of(
                CommunicationEventVersionStatus.READY,
                CommunicationEventVersionStatus.APPROVED,
                CommunicationEventVersionStatus.NEED_APPROVE
        ));
        return List.of(versionsRepository.getVersion(params.getEventId(), params.getIteration()));
    }

    protected List<CommunicationEventVersion> start(T params) {
        CommunicationEventVersion cev = versionsRepository.getVersion(params.getEventId(), params.getIteration());
        cev.setStatus(statusOnStart());
        versionsRepository.update(cev, CommunicationEventVersionStatus.APPROVED);
        return List.of(versionsRepository.getVersion(params.getEventId(), params.getIteration()));
    }

    protected CommunicationEventVersionStatus statusOnStart() {
        return CommunicationEventVersionStatus.READY;
    }

    protected List<CommunicationEventVersion> stop(T params) {
        CommunicationEventVersion cev = versionsRepository.getVersion(params.getEventId(), params.getIteration());
        cev.setStatus(statusOnStop());
        versionsRepository.update(cev, Set.of(
                CommunicationEventVersionStatus.ACTIVE,
                CommunicationEventVersionStatus.ACTIVATING
        ));
        return List.of(versionsRepository.getVersion(params.getEventId(), params.getIteration()));
    }

    protected CommunicationEventVersionStatus statusOnStop() {
        return CommunicationEventVersionStatus.NEED_ABORT;
    }

    protected void fillVersionModelByInput(
            CommunicationEventVersion cev,
            T params,
            boolean processInternal) {
        cev.setEventId(params.getEventId());
        if (params.getStartEventTime() != null) {
            cev.setStartTime(params.getStartEventTime());
        }
        if (params.getExpired() != null) {
            cev.setExpired(params.getExpired());
        }
        if (params.getTitle() != null) {
            cev.setTitle(params.getTitle());
        }
        if (params.getText() != null) {
            cev.setText(params.getText());
        }
        if (params.getButtonText() != null) {
            cev.setButtonText(params.getButtonText());
        }
        if (params.getButtonHref() != null) {
            cev.setButtonHref(params.getButtonHref());
        }
        if (params.getImageHref() != null) {
            cev.setImageHref(params.getImageHref());
        }
    }
}
