package ru.yandex.direct.internaltools.tools.communication;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.communication.model.CommunicationEvent;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolProcessingException;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationConfigurationAction;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationConfigurationParameters;
import ru.yandex.direct.internaltools.tools.communication.repository.CommunicationEventConfigurationRepository;
import ru.yandex.direct.internaltools.tools.communication.service.CommunicationConfigurationService;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Tool(
        name = "Управление конфигурациями событий",
        label = "communication_configurations",
        description = "Позволяет управлять файлами конфигураций событий",
        consumes = CommunicationConfigurationParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.COMMUNICATION_PLATFORM)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class CommunicationConfigurationTool implements BaseInternalTool<CommunicationConfigurationParameters> {
    private static final Logger logger = LoggerFactory.getLogger(CommunicationConfigurationTool.class);

    public static final Set<String> APPROVERS = Set.of(
            "yndx-a-dubov",
            "yndx-r-shakirova-developer",
            "yndx-arevolution"
    );

    private final Map<Class, CommunicationConfigurationService> services;
    private final CommunicationEventConfigurationRepository configurationRepository;

    @Autowired
    public CommunicationConfigurationTool(List<CommunicationConfigurationService<?>> services,
                                          CommunicationEventConfigurationRepository configurationRepository) {
        this.services = services.stream().collect(Collectors.toMap(CommunicationConfigurationService::getType,
                it -> it));
        this.configurationRepository = configurationRepository;
    }

    @Override
    public ValidationResult<CommunicationConfigurationParameters, Defect> validate(
            CommunicationConfigurationParameters params
    ) {
        var events = params.getIteration() == null
                ? configurationRepository.getEvents(params.getEventId())
                : configurationRepository.getEventVersions(params.getEventId(), params.getIteration());

        ItemValidationBuilder<CommunicationConfigurationParameters, Defect> builder = ItemValidationBuilder.of(params);
        if (params.getAction() == CommunicationConfigurationAction.ARCHIVE) {
            builder.item(events.getValue(), "events").check(isNull());
        } else if (params.getAction() != CommunicationConfigurationAction.SHOW_CURRENT) {
            builder.item(events.getValue(), "events").check(notNull());
        }
        if (builder.getResult().hasAnyErrors()) {
            var errors = builder.getResult().flattenErrors()
                    .stream()
                    .map(DefectInfo::toString)
                    .collect(Collectors.joining("\n"));
            logger.error("Validation error! {}", errors);
        }
        return builder.getResult();
    }

    @Override
    public InternalToolResult process(CommunicationConfigurationParameters params) {
        var expectedType = params.getIteration() == null
                ? CommunicationEvent.class
                : CommunicationEventVersion.class;
        CommunicationConfigurationService<?> service = services.get(expectedType);
        var result = service.process(params);
        if (result.isSuccessful()) {
            return new InternalToolMassResult<>(List.of(result.getResult()));
        } else {
            String errors;
            if (result.getValidationResult().getValue() != null) {
                errors = result.getValidationResult().getValue().toString();
            } else {
                errors = result.getErrors().stream().map(JsonUtils::toJson).collect(Collectors.joining("\n"));
            }
            throw new InternalToolProcessingException("Failed to perform " + params.getAction() + ", errors: " + errors);
        }
    }
}
