package ru.yandex.direct.internaltools.tools.communication;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventType;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolProcessingException;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventTypesAction;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventTypesParameters;
import ru.yandex.direct.internaltools.tools.communication.model.CommunicationEventTypesResponse;
import ru.yandex.direct.internaltools.tools.communication.service.CommunicationEventTypesService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus.ACTIVE;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus.ARCHIVED;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventStatus.NEW_;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventType.POPUP;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventType.RECOMMENDATION;
import static ru.yandex.direct.core.entity.communication.model.CommunicationEventType.SINGLE_LAUNCH;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "Управление списками коммуникационных событий",
        label = "communication_event_types",
        description = "Список актуальных видов событий. " +
                "Позволяет создать новый вид или заархивировать существующий (при условии, что все его итерации " +
                "заархивированы)",
        consumes = CommunicationEventTypesParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.COMMUNICATION_PLATFORM)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class CommunicationEventTypesTool implements BaseInternalTool<CommunicationEventTypesParameters> {
    private final CommunicationEventTypesService communicationEventTypesService;

    public CommunicationEventTypesTool(CommunicationEventTypesService communicationEventTypesService) {
        this.communicationEventTypesService = communicationEventTypesService;
    }

    @Override
    public ValidationResult<CommunicationEventTypesParameters, Defect> validate(CommunicationEventTypesParameters params) {
        var vb = ItemValidationBuilder.of(params, Defect.class);
        switch (params.getAction()) {
            case ADD:
            case ADD_RECOMMENDATION:
            case ADD_POPUP:
                if (params.getEventName() == null) {
                    throw new InternalToolValidationException("Name can't be null");
                }
                vb.item(params.getEventName(), "name of the event").check(notBlank());
                vb.item(params.getParsedOwnerLogins(), "logins of the owners")
                        .check(communicationEventTypesService.loginsExist());
                return vb.getResult();
            case UPDATE:
            case ARCHIVE:
                String message = communicationEventTypesService.checkChosenEvent(params);
                if (message != null) {
                    throw new InternalToolValidationException(message);
                }
                return vb.getResult();
            default:
                return vb.getResult();
        }
    }

    @Override
    public InternalToolMassResult<CommunicationEventTypesResponse> processWithoutInput() {
        var allEvents = communicationEventTypesService.getSortedEventsByTypesAndStatuses(
                List.of(SINGLE_LAUNCH, RECOMMENDATION, POPUP),
                List.of(NEW_, ACTIVE));
        return new InternalToolMassResult<>(communicationEventTypesService.convertToMassResponse(allEvents));
    }

    @Override
    public InternalToolMassResult<CommunicationEventTypesResponse> process(CommunicationEventTypesParameters params) {
        switch (params.getAction()) {
            case ADD:
            case ADD_RECOMMENDATION:
            case ADD_POPUP:
                var ownersWithOperators = params.getParsedOwnerLogins();
                ownersWithOperators.add(params.getOperator().getLogin());
                CommunicationEventType eventType = SINGLE_LAUNCH;
                if (params.getAction().equals(CommunicationEventTypesAction.ADD_RECOMMENDATION)) {
                    eventType = RECOMMENDATION;
                } else if (params.getAction().equals(CommunicationEventTypesAction.ADD_POPUP)) {
                    eventType = POPUP;
                }
                var recordedEvent = communicationEventTypesService
                        .addNewEventWithDefaults(
                                eventType,
                                params.getEventName(),
                                ownersWithOperators);
                return new InternalToolMassResult<>(communicationEventTypesService
                        .convertToMassResponse(singletonList(recordedEvent)
                ));
            case UPDATE:
                var owners = params.getParsedOwnerLogins();
                owners.add(params.getOperator().getLogin());
                var event =
                        communicationEventTypesService.updateOwnersAndGetChosenCommunicationEvent(params.getChosenEventId(), owners);
                if (event == null) {
                    throw new InternalToolProcessingException("Could not find communication event in the database");
                }
                return new InternalToolMassResult<>(communicationEventTypesService.convertToMassResponse(
                        singletonList(event))
                );
            case DISPLAY:
                List<CommunicationEventStatus> checkedStatuses = new ArrayList<>();
                if (params.getShowActive()) {
                    checkedStatuses.add(ACTIVE);
                }
                if (params.getShowNew()) {
                    checkedStatuses.add(NEW_);
                }
                if (params.getShowArchived()) {
                    checkedStatuses.add(ARCHIVED);
                }
                var shownEvents = communicationEventTypesService.getSortedEventsByTypesAndStatuses(
                        List.of(SINGLE_LAUNCH, RECOMMENDATION, POPUP),
                        checkedStatuses);

                return new InternalToolMassResult<>(communicationEventTypesService.convertToMassResponse(shownEvents));
            case ARCHIVE:
                var updatedEvent =
                        communicationEventTypesService.archiveAndGetChosenCommunicationEvent(params.getChosenEventId());
                if (updatedEvent == null) {
                    throw new InternalToolProcessingException("Could not find communication event in the database");
                }
                return new InternalToolMassResult<>(communicationEventTypesService.convertToMassResponse(
                        singletonList(updatedEvent))
                );
            default:
                return processWithoutInput();
        }
    }
}
